/* mirrirdir.c - duplicate two directories to every detail
   Copyright (C) 1998 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#define _MIRROR_DIR_C 1

#include "mostincludes.h"
#include <stdarg.h>
#include <signal.h>
#include "mirrordir.h"
#include "vfs/vfs.h"
#include "cmdlineopt.h"
#include "netrc.h"
#include "mad.h"

#ifdef HAVE_GETTIMEOFDAY
#define NICE_MESS "[--nice <num>]"
#else
#define NICE_MESS " "
#endif

#define COPY_BUF_SIZE 65536
#define UNDELETABLE_FILE "--keep-me"

char *control_dir = "/";
char *mirror_dir = 0;
char **control_files = 0;
char *progname;

int interupted = 0;
int terminate = 0;

/* options - see usage */
int verbose = 0;
int access_times = 0;
int always_write = 0;
int only_erase = 0;
int rest_access = 0;
#ifdef HAVE_GETTIMEOFDAY
int nice_to_cpu = 0;
#endif
int only_read = 1;
int strict_mtimes = 0;
int mtime_threshold = 0;
int no_mtimes = 0;
int dont_remove = 0;
int only_test = 0;
int handle_hardlinks = 1;
int erase_directories = 1;
int show_help = 0;
int show_version = 0;
char *backup_extension = 0;
int backup_howmany = 0;
int copy_mode = 0;
int allow_chmod = 1;
int allow_chown = 1;
int allow_netrc = 0; /*EK*/
int ignore_size = 0;
extern int ftpfs_option_allow_empty_directories;

char *ftpfs_password = 0;
unsigned long allow_permissions_mask = 0xFFFFFFFFL;
time_t outdate_time = 0;

char **control_mirror_directories = 0;

void *free_me[256];
unsigned char free_num = 0;

/* internal errno */
#define DIRECTORY_NOT_CREATED 1
int mirrordir_errno = 0;

/* updated constantly */
time_t current_time;

char *start_file = 0;

#define UNITS_BYTES	'B'
#define UNITS_KILO	'K'
#define UNITS_MEGA	'M'
#define UNITS_GIGA	'G'
int sum_units;

unsigned long block_size = 0;

unsigned long total_blocks = 0;
unsigned long blocks_written = 0;
unsigned long blocks_allowed = 0;

#define BLOCK_NOT_USED			-1
#define BLOCK_BEFORE_START		0
#define BLOCK_IN_RANGE			1
#define BLOCK_FULL			2

int block_pos = BLOCK_NOT_USED;


int file_in_use = 0;
int real_bad_error = 0;
int ignore_next_exclude = 0;

struct directory_list {
    struct stat stat;
    char *name;
    struct directory_list *next;
    struct directory_list *prev;
};

struct exclude_list {
    char *name;
#define EXCLUDE_REGEXP		1
#define EXCLUDE_GLOB		2
#define EXCLUDE_LITERAL		3
    int type;
    int ignore;		/* file/path not to be excluded, but to be overlooked completely */
    regex_t regexp;
    struct exclude_list *next;
    struct exclude_list *prev;
} *exclude_list = 0;

struct directory_entry {
    struct stat stat;
    char *name;
};

#ifndef HAVE_STRDUP
char *strdup (const char *s)
{
    char *p;
    p = malloc (strlen (s) + 1);
    if (!p)
	return 0;
    strcpy (p, s);
    return p;
}
#endif

/* {{{ reporting */

void progmess (char *s, char *par)
{
    fprintf (stderr, "%s: %s: %s\n", progname, s, par);
}

void verbmess (char *s, char *par)
{
    if (par)
	fprintf (stdout, "%s: ---verbose--- %s: %s\n", progname, s, par);
    else
	fprintf (stdout, "%s: ---verbose--- %s\n", progname, s);
}

void infomess (char *s, char *par)
{
    fprintf (stdout, "%s: %s: %s\n", progname, s, par);
}

void progmess_strerror (char *s, char *p)
{
    real_bad_error++;
#ifdef HAVE_STRERROR
    fprintf (stderr, "%s: %s: %s: %s\n", progname, s, p, strerror (mc_errno));
#else
    fprintf (stderr, "%s: %s: %s: %d\n", progname, s, p, mc_errno);
#endif
}

/* }}} reporting */

void be_nice_to_cpu (void)
{
#ifdef HAVE_GETTIMEOFDAY
    static long double s1 = 0.0, s2;
    struct timeval t;

    if (!nice_to_cpu)
	return;

    gettimeofday (&t, 0);
    s2 = (long double) t.tv_usec / 1000000.0 + (long double) t.tv_sec;

    if (s1 > 0.0) {
	s2 = (long double) (s2 - s1) * 1000000.0 * (long double) nice_to_cpu;
	if (verbose > 1) {
	    char s[12];
	    sprintf (s, "%d", (int) s2);
	    verbmess ("sleeping microseconds", s);
	}
	usleep ((unsigned long) s2);
    }
    gettimeofday (&t, 0);
    s1 = (long double) t.tv_usec / 1000000.0 + (long double) t.tv_sec;
#endif
}

/* result must be free'd, path of "" is treated as "/" */
char *join (char *path, char *file)
{
    char *r;
    int l;

    if (!*path)
	path = "/";
    while (*file == '/')
	file++;

    l = strlen (path);

    r = malloc (l + strlen (file) + 2);
    strcpy (r, path);
    if (path[l - 1] != '/')
	strcat (r, "/");
    strcat (r, file);
    return r;
}

void check_interrupt (void);

void check_keep_me (char *name)
{
    if (!*name)
	return;
    if (!strcmp (name, UNDELETABLE_FILE) || !strcmp (name + 1, UNDELETABLE_FILE)) {
	progmess ("aborting on finding a directory containing an undeletable file", name);
	exit (1);
    }
}

/* returns linked list of directory entries */
struct directory_list *read_list (char *path)
{
    struct directory_list *dl = 0, *next;
    DIR *dir;
    struct dirent *r;

    dir = mc_opendir (path);
    if (!dir) {
	progmess_strerror ("unable to open directory", path);
	return 0;
    }
    dl = malloc (sizeof (struct directory_list));
    memset (dl, 0, sizeof (struct directory_list));
    dl->name = strdup ("..");

    do {
	int l;
	r = mc_readdir (dir);
	check_interrupt ();
	next = malloc (sizeof (struct directory_list));
	memset (next, 0, sizeof (struct directory_list));
	next->prev = dl;
	if (dl)
	    dl->next = next;
	dl = next;
	if (r) {
	    l = NAMLEN (r);
	    dl->name = malloc (l + 1);
	    memcpy (dl->name, r->d_name, l);
	    dl->name[l] = '\0';
	}
    } while (r);

/* go to start of list, list will not be empty because of . and .. */
    rewind (dl);

    for (;;) {
	if (dl->name) {
	    if (strcmp ("..", dl->name) && strcmp (".", dl->name)) {
		char *p;
		p = join (path, dl->name);
		if (mc_lstat (p, &dl->stat)) {
		    progmess_strerror ("error trying to stat file", p);
		    free (dl->name);
		    dl->name = 0;
		}
		check_interrupt ();
		free (p);
	    } else {
		free (dl->name);
		dl->name = 0;
	    }
	}
	if (!dl->next)
	    break;
	dl = dl->next;
    }

    rewind (dl);

    mc_closedir (dir);

    return dl;
}

void free_list (struct directory_list *dl)
{
    if (!dl)
	return;
    rewind (dl);
    while (dl) {
	struct directory_list *next;
	next = dl->next;
	if (dl->name)
	    free (dl->name);
	free (dl);
	dl = next;
    }
}

void remove_list_entry (struct directory_list *dl)
{
    if (!dl->prev) {
	dl->next->prev = 0;	/* begining of list */
    } else if (!dl->next)
	dl->prev->next = 0;	/* end of list */
    else {
	dl->prev->next = dl->next;
	dl->next->prev = dl->prev;
    }
    if (dl->name)
	free (dl->name);
    dl->name = 0;
    free (dl);
}

struct directory_list *find_list_entry (struct directory_list *dl, char *name)
{
    rewind (dl);
    while (dl) {
	if (dl->name)
	    if (!strcmp (dl->name, name))
		return dl;
	dl = dl->next;
    }
    return 0;
}

void remove_name_entry (struct directory_list *dl, char *name)
{
    dl = find_list_entry (dl, name);
    if (dl)
	remove_list_entry (dl);
}

/* returns non-zero on error */
void create_dir (char *path, struct stat *stat)
{
    if (verbose)
	verbmess ("creating directory", path);
    if (only_test) {
	mirrordir_errno = DIRECTORY_NOT_CREATED;
    } else {
	if (mc_mkdir (path, stat->st_mode))
	    progmess_strerror ("error trying create directory", path);
	if (allow_chown)
	    if (mc_chown (path, stat->st_uid, stat->st_gid))
		progmess_strerror ("error trying chown/chgrp directory", path);
	if (allow_chmod)
	    if (mc_chmod (path, stat->st_mode))
		progmess_strerror ("unable to chmod directory", path);
    }
}

int remove_file_quiet (char *path)
{
    if (!only_test) {
	if (mc_unlink (path)) {
	    progmess_strerror ("unable to remove file", path);
	    return 1;
	}
    }
    return 0;
}

int remove_file (char *path)
{
    if (verbose)
	verbmess ("removing file", path);
    return remove_file_quiet (path);
}

/* result must NOT be free'd */
char *add_back_extension (char *path, int i)
{
    static char *r = 0;
    int l;
    if (r)
	free (r);
    l = strlen (path);
    r = malloc (l + strlen (backup_extension) + 16);
    memcpy (r, path, l);
    sprintf (r + l, backup_extension, i);
    return r;
}

int remove_dir (char *path);

static inline void utime_check (time_t modified, time_t access, char *path, int force)
{
    struct utimbuf t;
    if (!only_test || force) {
	t.actime = access;
	t.modtime = modified;
	if (mc_utime (path, &t))
	    progmess_strerror ("unable to change mtime/atime of file", path);
    }
}

/* BUG: this does not report when files cannot be removed */
int rename_file_dir (char *path, struct directory_list *dl)
{
    char *p;
    struct directory_list *d;
    int i;
    p = add_back_extension (path, backup_howmany);
    if (verbose)
	verbmess ("renaming file/dir", path);
    if ((d = find_list_entry (dl, strrchr (p, '/') + 1))) {
	if (S_ISDIR (d->stat.st_mode)) {
	    if (remove_dir (p))
		return 1;
	} else {
	    if (remove_file (p))
		return 1;
	}
    }
    if (backup_howmany > 1)
	for (i = backup_howmany; i > 1; i--) {
	    p = strdup (add_back_extension (path, i - 1));
	    if ((d = find_list_entry (dl, strrchr (p, '/') + 1))) {
		if (!only_test)
		    if (mc_rename (p, add_back_extension (path, i))) {
			progmess_strerror ("error trying to rename file/dir", p);
			free (p);
			return 1;
		    }
	    }
	    free (p);
	}
    if (!only_test) {
	p = add_back_extension (path, 1);
	if (is_hardlink (&dl->stat)) {
	    copy_regular_file (path, p, &dl->stat);
	    if (remove_file_quiet (path))
		return 1;
	} else if (mc_rename (path, p)) {
	    progmess_strerror ("error trying to rename file/dir", p);
	    return 1;
	}
    }
    time (&current_time);
    utime_check (current_time, current_time, p, 0);
    return 0;
}

int is_a_backup_file_name (char *name)
{
    char *p, *q;
    if (!backup_extension)
	return 0;
    p = name + strlen (name);
    q = backup_extension + strlen (backup_extension);
    while ((unsigned long) p > (unsigned long) name) {
	p--;
	q--;
	if (*q == 'd') {
	    if ((unsigned long) q > (unsigned long) backup_extension) {
		if (*(q - 1) == '%') {
		    int i;
		    q--;
		    if (!this_is_a_digit (*p))
			return 0;
		    while (this_is_a_digit (*p)) {
			if ((unsigned long) p == (unsigned long) name)
			    return 0;
			p--;
		    }
		    p++;
		    i = atoi (p);
		    if (i <= 0 || i > backup_howmany)
			return 0;
		} else if (*q != *p)
		    return 0;
	    } else if (*q != *p)
		return 0;
	} else if (*q != *p)
	    return 0;
	if ((unsigned long) q == (unsigned long) backup_extension)
	    return 1;
    }
    return 0;
}

int is_an_old_backup (struct stat *s)
{
    if (outdate_time) {
	time (&current_time);
	if ((time_t) s->st_mtime + outdate_time < (time_t) current_time)
	    return 1;
	else
	    return 0;
    }
    return 0;
}

/* removes all files or directories in a list */
int remove_list (char *path, struct directory_list *dl)
{
    int r = 0;
    rewind (dl);
    for (;;) {
	if (dl->name)
	    check_keep_me (dl->name);
	if (!dl->next)
	    break;
	dl = dl->next;
    }
    for (;;) {
	if (dl->name) {
	    int old, bname;
	    old = is_an_old_backup (&dl->stat);
	    bname = is_a_backup_file_name (dl->name);
	    if (!(!old && bname)) {
		char *p;
		p = join (path, dl->name);
		if (backup_extension && !bname)
		    if (rename_file_dir (p, dl))
			return 1;
		if (S_ISDIR (dl->stat.st_mode))
		    r += remove_dir (p);
		else
		    r += remove_file (p);
		free (p);
	    }
	}
	if (!dl->prev)
	    break;
	dl = dl->prev;
    }
    return r;
}

int remove_dir (char *path)
{
    struct directory_list *dl;
    int r = 0;
    dl = read_list (path);
    if (!dl)
	return 1;
    rewind (dl);
    if (dl->next->next)	/* list has only two entries : `.' and `..' */
	if (!erase_directories) {
	    progmess ("aborting on trying to erase a non-empty directory", path);
	    exit (1);
	}
    r = remove_list (path, dl);
    free_list (dl);
    if (verbose)
	verbmess ("removing directory", path);
    if (!only_test) {
	if (mc_rmdir (path)) {
	    progmess_strerror ("unable to remove directory", path);
	    r++;
	}
    }
    return r;
}

void set_vfs_temp_dir (char *p, char *q)
{
    char *d;
    *(strrchr ((d = strdup (q)), '/') + 1) = '\0';	/* there will always be a / in a full path */
    mc_setctl (p, MCCTL_SETTMPDIR, d);
    free (d);
}

void copy_regular_file (char *p, char *q, struct stat *stat)
{
    int f = 0, g = 0;
    char buf[COPY_BUF_SIZE];
    off_t togo;

    check_interrupt ();

    set_vfs_temp_dir (p, q);

    if (verbose) {
	if (only_erase)
	    verbmess ("truncating file", q);
	else
	    verbmess ("copying file", q);
    }
    if (!only_test) {
	if (only_read)
	    f = mc_open (p, O_RDONLY);
	else
	    f = mc_open (p, O_RDWR);	/* open for reading AND writing */
    }
/*
   The reason we open for reading and writing is that if a file is
   in the process of being written, we don't want to copy it in its
   unclean state. If another process is writing to the file, the open
   will return an error and we are safe.
 */
    if (f < 0) {
	if (only_read)
	    progmess_strerror ("unable to open control file for reading", p);
	else
	    progmess_strerror ("unable to open control file for reading and writing", p);
	if (!only_read) {
	    file_in_use++;
	    real_bad_error++;
	}
	return;
    }
    if (!only_test) {
	g = mc_open (q, O_CREAT | O_WRONLY | O_TRUNC, stat->st_mode);
    }
    if (g < 0) {
	progmess_strerror ("unable to open file for writing", q);
	mc_close (f);
	return;
    }
    togo = stat->st_size;

    if (!only_test && !only_erase) {
	while (togo > 0) {
	    off_t count, c;
	    char *b;
	    be_nice_to_cpu ();
	    count = mc_read (f, buf, COPY_BUF_SIZE);
	    if (count < 0) {
		progmess_strerror ("error trying to read from file", p);
		mc_close (f);
		mc_close (g);
		return;
	    }
	    b = buf;
	    togo -= count;
	    while (count > 0) {
		c = mc_write (g, b, count);
		if (c < 0) {
		    progmess_strerror ("error trying to write to file", q);
		    mc_close (f);
		    mc_close (g);
		    return;
		}
		b += c;
		count -= c;
	    }
	}

	mc_close (f);
	if (mc_close (g))
	    progmess_strerror ("error trying to close file", q);
	mc_setctl (p, MCCTL_REMOVELOCALCOPY, 0);
	mc_setctl (q, MCCTL_REMOVELOCALCOPY, 0);

/* chown comes first because it changes the sticky bit */
	if (allow_chown)
	    if (mc_chown (q, stat->st_uid, stat->st_gid))
		progmess_strerror ("error trying chown/chgrp regular file", q);
/* now reset the permissions in case the sticky bit has changed */
	if (allow_chmod)
	    if (mc_chmod (q, stat->st_mode))
		progmess_strerror ("unable to chmod regular file", q);
    }
    if (rest_access)
	utime_check (stat->st_mtime, stat->st_atime, p, 1);
    utime_check (stat->st_mtime, stat->st_atime, q, 0);
    check_interrupt ();
}

int copy_symlink (char *p, char *q, int force)
{
    char pbuf[MAX_PATH_LEN + 1], qbuf[MAX_PATH_LEN + 1];
    int pl, ql;
    pl = mc_readlink (p, pbuf, MAX_PATH_LEN);
    if (pl < 0) {
	progmess_strerror ("error trying read symlink", p);
	return 0;
    }
    pbuf[pl] = '\0';
    if (!force) {
	ql = mc_readlink (q, qbuf, MAX_PATH_LEN);
	if (ql < 0) {
	    progmess_strerror ("error trying read symlink", q);
	    return 0;
	}
	qbuf[ql] = '\0';
	force = strcmp (pbuf, qbuf);
	if (force) {
	    if (only_erase && verbose)
		verbmess ("removing symlink", q);
	    if (remove_file_quiet (q))
		return 1;
	}
    }
    if (force) {
	if (verbose && !only_erase)
	    verbmess ("creating symlink", q);
	if (!only_test && !only_erase) {
	    if (mc_symlink (pbuf, q))
		progmess_strerror ("error trying create symlink", q);
	    return 1;
	}
    }
    return 0;
}

void make_special (char *path, struct stat *stat)
{
    if (verbose)
	verbmess ("creating device file", path);
    if (!only_test && !only_erase) {
	if (mc_mknod (path, stat->st_mode, stat->st_rdev))
	    progmess_strerror ("error trying create device file", path);
    }
/* mknod does not seem to set user and group permissions */
/* chown comes first because it changes the sticky bit */
    if (!only_test && !only_erase) {
	if (allow_chown)
	    if (mc_chown (path, stat->st_uid, stat->st_gid))
		progmess_strerror ("error trying chown/chgrp device file", path);
	if (allow_chmod)
	    if (mc_chmod (path, stat->st_mode))
		progmess_strerror ("unable to chmod device file", path);
    }
    utime_check (stat->st_mtime, stat->st_atime, path, 0);
}

/* returns zero if the directory in which the file was listed was not modified */
int copy_file (char *p, char *q, struct stat *p_stat, struct stat *q_stat)
{
    mode_t m = p_stat->st_mode;
    if (S_ISREG (m)) {
	if (only_erase && !q_stat)
	    return 0;
	copy_regular_file (p, q, p_stat);
    } else if (S_ISLNK (m))
	return copy_symlink (p, q, (q_stat == 0));	/* hack to not read a symlink if it does not yet exist on the mirror side */
    else if (S_IS_SPECIAL (m))
	make_special (q, p_stat);
    else {
	real_bad_error++;
	progmess ("what the hell kind of file is this?", p);
    }
    return 0;
}

static inline int time_out_range (time_t tcontrol, time_t tmirror)
{
    return tmirror < tcontrol - mtime_threshold || tmirror > tcontrol + mtime_threshold;
}

int compare_times (time_t tcontrol, time_t tmirror)
{
    if (always_write)
	return 1;
    if (no_mtimes)
	return 0;
    if (strict_mtimes)
	return time_out_range (tcontrol, tmirror);
    return tmirror < tcontrol - mtime_threshold;	/* i.e. the mirror file is *older* than the control file */
}

/* returns zero if the directory in which the file was listed was not modified */
int copy_hardlink_or_file (char *p, char *q, struct stat *p_stat, struct stat *q_stat)
{
    int r = 0;
    if (is_hardlink (p_stat)) {
	if (find_hardlink (p_stat)) {
	    if (!q_stat) {	/* the mirror file not existant */
		if (verbose)
		    verbmess ("copying file as hardlink", q);
		create_link (q);
		reduce_hardlink ();
		return 1;	/* directory modified */
	    } else if (!correct_link (q_stat)) {
/* the file has no hardlinks or is linked to the wrong place */
		if (remove_file_quiet (q))
		    return 1;
		if (verbose)
		    verbmess ("overwriting file as hardlink", q);
		create_link (q);
		reduce_hardlink ();
		return 1;	/* directory modified */
	    }
	    reduce_hardlink ();
	    return 0;		/* directory not modified */
	}
    }
    r |= copy_file (p, q, p_stat, q_stat);
/* if we got to here, then the hardlink is not found in the list */
    if (is_hardlink (p_stat)) {
	struct stat s;
	if (!q_stat) {
	    r = 1;
	    mc_stat (q, &s);
	    q_stat = &s;
	}
	add_hardlink (q, p_stat, q_stat);	/* must be comming accross this family of links
						   for the first time, so create a new list entry */
    }
    return r | !q_stat;		/* directory not modified */
}

int is_special_prefix (char *path);

/*
   remove() can remove anything except a directory, hence the reasoning
   in separating the file operations from the directory operations at a
   high level. Returns 1 if a modification was made that would change the
   modified time of the directory.
 */
int duplicate (char *p, char *q, struct directory_entry *d, struct directory_list *list)
{
    mode_t m = d->stat.st_mode;

    if (list)
	if (!is_hardlink (&d->stat) && is_hardlink (&list->stat)) {
	    if (verbose > 1)
		verbmess ("hardlink changed to non-hardlink", q);
	    list = 0;
	    remove_file_quiet (q);
	}

    if (!list) {
	/* the file or directory was not found in the mirror list */
	if (S_ISDIR (m)) {
	    if (only_erase)
		return 0;
	    create_dir (q, &d->stat);
	    return 1;
	} else {
	    if (verbose > 1)
		verbmess ("creating new file", q);
	    copy_hardlink_or_file (p, q, &d->stat, 0);
	    return 1;
	}
    }				/* else { */
    /* check if a directory has become a file and is hence in the way */
    if (S_ISDIR (list->stat.st_mode) && !S_ISDIR (m)) {
	if (verbose > 1)
	    verbmess ("replacing directory with file", q);
	if (backup_extension)
	    if (rename_file_dir (q, list))
		return 1;
	if (remove_dir (q))	/* recursively */
	    return 1;
	copy_hardlink_or_file (p, q, &d->stat, 0);
	return 1;
    }
    if (S_ISREG (m) || S_IS_SPECIAL (m)) {
	int r = 0, copy = 0, add_hlink = 0;
	if ((S_IFMT & m) != (S_IFMT & list->stat.st_mode)) {
	    copy = 1;
	    r = 1;
	    if (remove_file_quiet (q))
		return 1;
	    if (S_IS_SPECIAL (m))
	    	if (verbose > 1)
		    verbmess ("updating device file that has changed type", q);
	}
/* regular files must be rewritten if they have been modified */
	if (S_ISREG (m))
	    if (compare_times (d->stat.st_mtime, list->stat.st_mtime) || (d->stat.st_size != list->stat.st_size && !ignore_size)) {
		if (verbose > 1)
		    verbmess ("updating modified file", q);
		if (is_special_prefix (q)) {	/* with a local fs, rewriting a file is possible - not so with ftp so we remove it first */
		    r = 1;
		    if (remove_file_quiet (q))
			return 1;
		}
		copy = 1;
	    }

/* ...or if they are linked to the wrong file */
	if (is_hardlink (&d->stat)) {
	    if (find_hardlink (&d->stat)) {
		if (correct_link (&list->stat))
		    copy = 0;	/* a hardlink would have already been copied - also no need to back it up */
		else
		    copy = 1;
	    } else {
		add_hlink = 1;
	    }
	}
	if (copy) {
	    if (S_ISREG (m) && backup_extension) {	/* do not backup device files */
		r = 1;
		if (rename_file_dir (q, list))
		    return 1;
		r |= copy_hardlink_or_file (p, q, &d->stat, 0);
	    } else {
		r |= copy_hardlink_or_file (p, q, &d->stat, &list->stat);
	    }
	    return r;		/* we can return now - add_hardlink would have been run by copy_hardlink above */
	} else if (is_hardlink (&d->stat)) {
	    if (add_hlink)
		add_hardlink (q, &d->stat, &list->stat);	/* must be coming accross this family of links
								   for the first time, so create a new list entry */
	    else
		reduce_hardlink ();
	}
    } else if (S_ISLNK (m)) {
	if ((S_IFMT & m) != (S_IFMT & list->stat.st_mode)) {
	    if (remove_file_quiet (q))
		return 1;
	    copy_hardlink_or_file (p, q, &d->stat, 0);
	    return 1;
	} else {
	    return copy_hardlink_or_file (p, q, &d->stat, &list->stat);
	}
    } else if (S_ISDIR (m)) {
/* check if a file has become a directory */
	if (!S_ISDIR (list->stat.st_mode)) {
	    if (backup_extension && S_ISREG (list->stat.st_mode))
		if (rename_file_dir (q, list))
		    return 1;
	    if (remove_file (q))
		return 1;
	    if (only_erase)
		return 0;
	    create_dir (q, &d->stat);
	    return 1;
	}
    } else {
	real_bad_error++;
	progmess ("what the hell kind of file is this?", p);
	return 0;
    }

/*
   If we get here, the mirror file/directory is of the same type,
   and just needs to have its permissions, ownership and times updated.
 */

    if (S_ISLNK (m))
	return 0;		/* no need to worry about links */
/* check if ownership is identical */
/* chown comes first because it changes the sticky bit */
    if (list->stat.st_uid != d->stat.st_uid || list->stat.st_gid != d->stat.st_gid) {
	if (verbose && allow_chown)
	    verbmess ("changing ownership", q);
	if (!only_test) {
	    if (allow_chown)
		if (mc_chown (q, d->stat.st_uid, d->stat.st_gid))
		    progmess_strerror ("unable to chown/chgrp dir/file", q);
	}
    }
/* check permissions */
    if (list->stat.st_mode != d->stat.st_mode) {
	if (verbose && allow_chmod)
	    verbmess ("changing permissions", q);
	if (!only_test) {
	    if (allow_chmod)
		if (mc_chmod (q, d->stat.st_mode))
		    progmess_strerror ("unable to chmod dir/file", q);
	}
    }
/* modified and access times */
    if (!S_ISDIR (m)) {		/* these must be handled seperately since access times are always changed */
	if ((time_out_range (list->stat.st_atime, d->stat.st_atime) && access_times) || time_out_range (list->stat.st_mtime, d->stat.st_mtime)) {
	    if (verbose)
		verbmess ("changing mtime/atime", q);
	    utime_check (d->stat.st_mtime, d->stat.st_atime, q, 0);
	}
    }
    return 0;
}

enum {
    PATH_INCLUDED,
    PATH_IGNORED,
    PATH_EXCLUDED,
};

int excluded (char *p)
{
    char *q;
    struct exclude_list *l;
    if (is_listed (p))
	return PATH_EXCLUDED;
    if (!exclude_list)
	return PATH_INCLUDED;
    rewind (exclude_list);
    q = strrchr (p, '/') + 1;	/* full paths, so there will always be a / somewhere */
    l = exclude_list;
    while (l) {
	int ignore;
	ignore = l->ignore ? PATH_IGNORED : PATH_EXCLUDED;
	switch (l->type) {
	case EXCLUDE_LITERAL:
	    if (!strcmp (p, l->name)) {		/* it only occurs once, so we can remove it */
		free (l->name);
		if (!l->prev && !l->next) {
		    exclude_list = 0;
		    free (l);
		    return ignore;
		}
		if (!l->prev) {
		    l->next->prev = 0;
		} else if (!l->next)
		    l->prev->next = 0;
		else {
		    l->prev->next = l->next;
		    l->next->prev = l->prev;
		}
		if ((unsigned long) l == (unsigned long) exclude_list)
		    exclude_list = l->next;
		free (l);
		return ignore;
	    }
	    break;
	case EXCLUDE_REGEXP:
	    if (!regexec (&l->regexp, p, 0, NULL, 0))
		return ignore;
	    break;
	case EXCLUDE_GLOB:
	    if (!regexec (&l->regexp, q, 0, NULL, 0))
		return ignore;
	    break;
	}
	l = l->next;
    }
    return PATH_INCLUDED;
}

unsigned long blocks_to_kilobytes (unsigned long blocks)
{
    if (block_size > 1024)
	return (unsigned long) blocks *(block_size / 1024);
    return (unsigned long) blocks / (1024 / block_size);
}

unsigned long bytes_to_blocks (off_t bytes)
{
    unsigned long file_blocks;
    file_blocks = bytes / block_size;
    if (bytes % block_size)
	file_blocks++;
    return file_blocks;
}

/* returns non-zero if in range */
int block_range (char *p, struct directory_entry *d)
{
    int r = 0;
    unsigned long file_blocks;

    file_blocks = bytes_to_blocks (d->stat.st_size);
    if (is_hardlink (&d->stat))
	if (find_hardlink (&d->stat))
	    file_blocks = 1;	/* this isn't true, but I have a feeling thats its a good idea */

    total_blocks += file_blocks;
    switch (block_pos) {
    case BLOCK_NOT_USED:
	blocks_written += file_blocks;
	r = 1;
	break;
    case BLOCK_BEFORE_START:
	r = 0;
	if (!start_file)
	    goto in_range;
	if (!strcmp (start_file, p)) {
	  in_range:
	    blocks_written += file_blocks;
	    r = 1;
	    block_pos = BLOCK_IN_RANGE;
	} else {
	    if (S_ISDIR (d->stat.st_mode)) {
		if (!strncmp (start_file, p, strlen (p)) && start_file[strlen (p)] == '/') {
		    blocks_written += file_blocks;
		    r = 1;
		}
	    }
	}
	break;
    case BLOCK_IN_RANGE:
	if (blocks_written + file_blocks <= blocks_allowed || !blocks_allowed) {
	    blocks_written += file_blocks;
	    r = 1;
	} else {
	    infomess ("filled up all blocks - first file/dir not mirrored", p);
	    block_pos = BLOCK_FULL;
	    r = 0;
	}
	break;
    case BLOCK_FULL:
	r = 0;
	break;
    }
    if (file_blocks >= blocks_allowed && blocks_allowed) {
	real_bad_error++;
	progmess ("single file to large to fit within total bytes", p);
    }
    if (verbose > 2) {
	char s[128];
	sprintf (s, "%s file/dir, consumes %lukB: running total %lukB: mirrored total %lukB", \
	r ? "included" : "excluded", blocks_to_kilobytes (file_blocks), \
		 blocks_to_kilobytes (total_blocks), blocks_to_kilobytes (blocks_written));
	verbmess (s, p);
    }
    return r;
}

int subdir_present (char *full_path, char *high_path, struct directory_list *dl)
{
    int l;
    char *r;
    char name[MAX_PATH_LEN];
    l = strlen (high_path);
    if (l > strlen (full_path))	/* not applicable */
	return 1;
    if (strncmp (full_path, high_path, l) || full_path[l] != '/')	/* not applicable */
	return 1;
    strcpy (name, full_path + l + 1);
    r = strchr (name, '/');
    if (r)
	*r = '\0';
    if (find_list_entry (dl, name))
	return 1;
    return 0;
}

/* returns non-zero if a modification was made that would change the modified time of the directory */
int recurs_dir (char *control, char *mirror,...)
{
    char **files = 0;

    va_list ap;
    int r = 0;
    struct directory_list *dl = 0;
    struct directory_list *dm;

    if (control) {
	dl = read_list (control);
    } else {
	va_start (ap, mirror);
	files = va_arg (ap, char **);
	va_end (ap);
    }
    if (start_file)
	if (!subdir_present (start_file, control, dl)) {
	    progmess ("starting file or one of its components was not found, aborting immediately", start_file);
	    exit (1);
	}
    dm = read_list (mirror);
    if ((control && !dl) || !dm)
	return -1;

    if (control)
	mc_setctl (control, MCCTL_SETDIRTIMEOUT, (void *) 3600);
    mc_setctl (mirror, MCCTL_SETDIRTIMEOUT, (void *) 3600);

    be_nice_to_cpu ();

    for (;;) {
	struct directory_list *next = 0;
	struct directory_list f;
	if (control) {
	    next = dl->next;
	} else {
	    dl = &f;
	    if (!*files)
		break;
	    if (mc_lstat (*files, &dl->stat)) {
		progmess_strerror ("error trying to stat file", *files);
		real_bad_error++;
		dl->name = 0;
	    } else {
		dl->name = strrchr (*files, '/') + 1;
	    }
	    dl->next = 0;
	    dl->prev = 0;
	}
	if (dl->name) {
	    char *p, *q;
	    int e;
	    check_interrupt ();
	    if (control) {
		p = join (control, dl->name);
	    } else {
		p = strdup (*files);
	    }
	    e = excluded (p);
	    if (e == PATH_IGNORED) {
		/* ignoring this entry, so remove it from the mirror list */
		remove_name_entry (dm, dl->name);
	    } else if (e == PATH_INCLUDED && (mode_t_to_unsigned_long (dl->stat.st_mode) & allow_permissions_mask)) {	/* excluded files must not count toward the running total of blocks */
		if (block_range (p, (struct directory_entry *) dl)) {
		    struct directory_list *d;
		    int m;
		    q = join (mirror, dl->name);
		    d = find_list_entry (dm, dl->name);
		    mirrordir_errno = 0;
		    m = duplicate (p, q, (struct directory_entry *) dl, d);
		    r += m;
		    if (S_ISDIR (dl->stat.st_mode)) {
			if (mirrordir_errno != DIRECTORY_NOT_CREATED) {
			    m = recurs_dir (p, q);
			    mc_setctl (p, MCCTL_FLUSHDIR, 0);	/* we aren't going to go in there again, so free memory */
			    mc_setctl (q, MCCTL_FLUSHDIR, 0);
			    if (m == -1) {
				real_bad_error++;
				progmess ("error trying to mirror directory", p);
				m = 0;
			    }
			    if (d) {
				if (S_ISDIR (d->stat.st_mode)) {
				    if (access_times || m || time_out_range (dl->stat.st_mtime, d->stat.st_mtime)) {
					if (verbose > 1 || (verbose && (m || \
									(time_out_range (dl->stat.st_atime, d->stat.st_atime) \
								      &&\
						       access_times) || \
									time_out_range (dl->stat.st_mtime, d->stat.st_mtime)) \
					    )) {
					    char mess[64] = "changing directory";
					    if (time_out_range (dl->stat.st_atime, d->stat.st_atime) && access_times)
						strcat (mess, " atime");
					    else if (verbose > 1)
						strcat (mess, " atime/mtime");
					    if (time_out_range (dl->stat.st_mtime, d->stat.st_mtime))
						strcat (mess, " mtime");
					    if (strlen (mess) > 20)	/* hack to never print a changing nothing message */
						verbmess (mess, q);
					}
					utime_check (dl->stat.st_mtime, dl->stat.st_atime, q, 0);
				    }
				} else {
				    utime_check (dl->stat.st_mtime, dl->stat.st_atime, q, 0);
				}
			    } else {
				utime_check (dl->stat.st_mtime, dl->stat.st_atime, q, 0);
			    }
			    r += m;
			}
			if (rest_access) {
			    if (verbose > 1)
				verbmess ("changing directory mtime/atime", p);
			    utime_check (dl->stat.st_mtime, dl->stat.st_atime, p, 1);
			}
		    }
		    remove_name_entry (dm, dl->name);	/* checked this entry, so remove it from the mirror list */
		    free (q);
		}
	    }
	    free (p);
	}
	if (control) {
	    if (!next)
		break;
	    dl = next;
	} else {
	    files++;
	}
    }

    if (!dont_remove)
	remove_list (mirror, dm);

    free_list (dm);
    if (control)
	free_list (dl);
    return r;
}


/* {{{ command line processing */

void usage (int exit_code)
{
    if (exit_code) {
	progmess ("mirrordir: error on command-line", "try `mirrordir --help'");
	exit (exit_code);
    }
    fprintf (stdout, "\
\n\
Usage:\n\
    mirrordir -[abBdDFGhklMmNOopRrsTtvVX] <control> <mirror>\n\
    mirrordir -c <path> [<path> ...] <dest>\n\
    copydir -[abBdeFGhklMmNOopRrsTtvVX] <path> [<path> ...] <dest>\n\
\n\
But usually just\n\
    mirrordir [--exclude <path>] <control> <mirror>\n\
    copydir <path> [<path> ...] <dest>\n\
\n\
Mirrordir makes a minimal set of changes to the directory <mirror> to\n\
make it identical to the directory <control>. Mirrordir dives into\n\
subdirectories recursively and duplicates all types of files exactly.\n\
Mirrordir is a DANGEROUS command because it deletes sub-directories\n\
recursively as required to make <mirror> identical to <control>.\n\
\n\
Copydir is equivalent to\n\
    mirrordir -c --no-erase-directories --keep-files ...\n\
\n\
\n\
  -c  --copy-mode               copy multiple files on the command line as\n\
                                allowed by cp - this is the default for\n\
                                copydir and implies --keep-files\n\
\n\
  -a, --access-times            duplicate even access times of <control>\n\
  -m, --strict-mtimes           copy files if modified times differ at all\n\
      --no-mtimes               do not look at the mtime of a file when\n\
                                trying to decide whether to copy it\n\
      --ignore-size             do not look at the size of a file when\n\
                                trying to decide whether to copy it\n\
  -d, --mtime-threshold <sec>   set the allowable error in in mtime\n\
  -A, --always-write            overwrite all, regardless of mtime or size\n\
  -r, --restore-access          restore access times of <control>\n\
      --no-chown                never set ownerships of a file\n\
      --no-chmod                never set permissions of a file\n\
  -D, --only-delete             do nothing that will consume more disk space\n\
\n\
  -e, --erase-directories       if a directory and file conflict, then erase\n\
                                the directory - this is the default for\n\
                                mirrordir\n\
      --no-erase-directories    if a directory and file conflict, then\n\
                                abort - this is the default for copydir\n\
\n\
  -b, --backup-extension <ext>  with this extension, backup changed files\n\
  -N, --num-backups <num>       backup levels, <ext> must have a %%d in it\n\
  -O, --backup-outdate <sec>    erase backup files older than <sec> seconds\n\
\n\
  -B, --block-size <bytes>      device's block size for space calculations\n\
  -M, --max-bytes <num>[K|M|G]  max bytes to mirror, then abort with filename\n\
  -s, --starting-file <path>    start mirroring only after reading <path>\n\
\n\
  [-i, --ignore-next-exclude]        the next exclude option means to leave\n\
                                     the path completely unchanged\n\
  [-i] -X, --exclude <path>          exclude full path matching <path>\n\
  -F, --exclude-from <file>          read list of files to exclude from <file>\n\
  [-i] -G, --exclude-glob <expr>     exclude file/dir names matching <expr>\n\
  [-i] -R, --exclude-regexp <expr>   exclude full paths matching <expr>\n\
\n\
  -T, --allow-matches <type>    only mirror files matching <type> symbolic\n\
                                expression.\n\
\n\
  -h, --help                    print help and exit\n\
  -v, --verbose                 give details of modifications being done\n\
  -V, --version                 print version and exit\n\
\n\
  -k, --keep-files              do not erase files unless they conflict - this\n\
                                is the default for copydir\n\
  -l, --no-hard-links           treat hardlinks as regular files\n\
  -L, --strict-locking          open control files for reading and writing\n\
  -o, --only-read               don't open control files for writing (default)\n\
\n\
  -p, --password <password>     ftp password\n\
      --netrc                   consult ~/.netrc (this is the default)\n\
      --no-netrc                don't consult ~/.netrc\n\
\n\
      --allow-empty-ftp-dirs    some servers cause an error with no . nor ..\n\
\n\
  -t, --dry-run, --test-only    output merely what would be done\n\
\n\
      --nice <num>              consume less cpu time, 0 < num << 50\n\
                                not supported on some systems.\n\
\n\
All paths may be URL's, but at the moment only ftp:// and mc:\n\
are supported.\n\
\n\
Please send comments, suggestions and bug reports to\n\
    mirrordir@mail.obsidian.co.za\n\
\n\
\n");
    exit (0);
}

char cwd[MAX_PATH_LEN + 1];

#ifdef HAVE_MAD
char *mad_pathdup (char *p, char *file, int line)
#else
char *pathdup (char *p)
#endif
{
    char *q, *r, *f = 0;

    if (!is_special_prefix (p)) {
	if (*p != '/') {
	    f = r = malloc (strlen (cwd) + strlen (p) + 2);
	    strcpy (r, cwd);
	    strcat (r, "/");
	    strcat (r, p);
	    p = r;
	}
    }
#ifdef HAVE_MAD
    r = q = mad_alloc (strlen (p) + 2, file, line);
#else
    r = q = malloc (strlen (p) + 2);
#endif
    if (is_special_prefix (p)) {
	while (*p != ':')
	    *q++ = *p++;
	*q++ = *p++;
	if (!strncmp (p, "//", 2)) {
	    *q++ = *p++;
	    *q++ = *p++;
	}
    }
    for (;;) {
	if (!*p) {
	    *q = '\0';
	    break;
	}
	if (*p != '/') {
	    *q++ = *p++;
	} else {
	    while (*p == '/') {
		*q = '/';
		if (!strncmp (p, "/./", 3) || !strcmp (p, "/."))
		    p++;
		else if (!strncmp (p, "/../", 4) || !strcmp (p, "/..")) {
		    p += 2;
		    *q = ' ';
		    q = strrchr (r, '/');
		    if (!q) {
			q = r;
			*q = '/';
		    }
		}
		p++;
	    }
	    q++;
	}
    }
/* get rid of trailing / */
    if (r[0] && r[1])
	if (*--q == '/')
	    *q = '\0';
    if (f)
	free (f);
    return r;
}

struct exclude_list *add_to_exclude_list (struct exclude_list *exclude_list, char *arg, int prepend_cwd)
{
    struct exclude_list *next;
    next = malloc (sizeof (struct exclude_list));
    memset (next, 0, sizeof (struct exclude_list));
    next->prev = exclude_list;
    if (exclude_list)
	exclude_list->next = next;
    exclude_list = next;
    if (prepend_cwd)
	exclude_list->name = pathdup (arg);
    else
	exclude_list->name = strdup (arg);
    return exclude_list;
}

/* result must be free'd */
char *glob_translate (char *s)
{
    char *g, *p;
    p = g = malloc (strlen (s) * 2 + 5);
    *p++ = '^';
    while (*s) {
	switch (*s) {
	case '*':
	    *p++ = '.';
	    *p = '*';
	    break;
	case '?':
	    *p = '.';
	    break;
	case '.':
	    *p++ = '\\';
	    *p = '.';
	    break;
	default:
	    *p = *s;
	    break;
	}
	p++;
	s++;
    }
    *p++ = '$';
    *p = '\0';
    return g;
}

int opt_password_callback (int a, char *s)
{
    ftpfs_password = strdup (s);
    memset (s, 0, strlen (s));
    return 0;
}

int opt_exclude_callback (int a, char *s)
{
    char *p;
    switch (a) {
    case 'X':
	exclude_list = add_to_exclude_list (exclude_list, s, 1);
	exclude_list->type = EXCLUDE_LITERAL;
	exclude_list->ignore = (ignore_next_exclude != 0);
	ignore_next_exclude = 0;
	break;
    case 'G':
	exclude_list = add_to_exclude_list (exclude_list, s, 0);
	p = glob_translate (exclude_list->name);
	free (exclude_list->name);
	exclude_list->name = p;
	exclude_list->type = EXCLUDE_GLOB;
	exclude_list->ignore = (ignore_next_exclude != 0);
	ignore_next_exclude = 0;
	if (regcomp (&exclude_list->regexp, exclude_list->name, REG_EXTENDED | REG_NOSUB)) {
	    progmess ("error converting glob expression to regular expression", s);
	    return 1;
	}
	break;
    case 'R':
	exclude_list = add_to_exclude_list (exclude_list, s, 0);
	exclude_list->type = EXCLUDE_REGEXP;
	exclude_list->ignore = (ignore_next_exclude != 0);
	ignore_next_exclude = 0;
	if (regcomp (&exclude_list->regexp, exclude_list->name, REG_EXTENDED | REG_NOSUB)) {
	    progmess ("error compiling regular expression", s);
	    return 1;
	}
	break;
    }
    return 0;
}

int calc_bytes (char *s)
{
    int c;

    if (!this_is_a_digit (s[0])) {
	progmess ("--max-bytes reguires an integer", s);
	return 1;
    }
    blocks_allowed = atol (s);
    c = s[strlen (s) - 1];
    switch (c) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
	sum_units = UNITS_BYTES;
	blocks_allowed /= block_size;
	break;
    case 'K':
    case 'k':
	sum_units = UNITS_KILO;
	blocks_allowed <<= 10;
	blocks_allowed /= block_size;
	break;
    case 'M':
    case 'm':
	sum_units = UNITS_MEGA;
	blocks_allowed <<= 20;
	blocks_allowed /= block_size;
	break;
    case 'G':
    case 'g':
	sum_units = UNITS_GIGA;
	blocks_allowed <<= 20;
	blocks_allowed /= block_size;
	blocks_allowed <<= 10;
	break;
    default:
	progmess ("unknown unit, only B, K, M, or G allowed", s);
	return 1;
    }
    if (blocks_allowed < 1) {
	progmess ("--max-bytes allowed must be more than 0", s);
	return 1;
    }
    block_pos = BLOCK_BEFORE_START;
    return 0;
}


int pars_opts (int argc, char **argv)
{
    int i;
    double outdate_time_float = 0.0;
    char *max_str = 0;
    char *exclude_file = 0;
    char *allow_str = 0;

    struct prog_options arg_opts[] =
    {
/* 0*/	{' ', "", "", ARG_STRINGS, 0, 0, 0},
/* 1*/	{'a', "", "--access-times", ARG_SET, 0, 0, &access_times},
/* 2*/	{'A', "", "--always-write", ARG_SET, 0, 0, &always_write},
/* 3*/	{'b', "", "--backup-extension", ARG_STRING, &backup_extension, 0, 0},
/* 4*/	{'B', "", "--block-size", ARG_INT, 0, 0, &block_size},
/* 5*/	{'c', "", "--copy-mode", ARG_SET, 0, 0, &copy_mode},
/* 6*/	{'d', "", "--mtime-threshold", ARG_INT, 0, 0, &mtime_threshold},
/* 7*/	{'D', "", "--only-delete", ARG_SET, 0, 0, &only_erase},
/* 8*/	{ 0 , "", "--no-erase-directories", ARG_CLEAR, 0, 0, &erase_directories},
/* 9*/	{'e', "", "--erase-directories", ARG_SET, 0, 0, &erase_directories},
/*10*/	{'F', "", "--exclude-from", ARG_STRING, /* &exclude_file */ 0, 0, 0},
/* 1*/	{'G', "", "--exclude-glob", ARG_CALLBACK, 0, 0, (void *) opt_exclude_callback},
/* 2*/	{'h', "", "--help", ARG_ADD, 0, 0, &show_help},
/* 3*/	{'k', "", "--keep-files", ARG_SET, 0, 0, &dont_remove},
/* 4*/	{'l', "", "--no-hard-links", ARG_CLEAR, 0, 0, &handle_hardlinks},
/* 5*/	{'M', "", "--max-bytes", ARG_STRING, /* &max_str */ 0, 0, 0},
/* 6*/	{'m', "", "--strict-mtimes", ARG_SET, 0, 0, &strict_mtimes},
/* 7*/	{'N', "", "--num-backups", ARG_INT, 0, 0, &backup_howmany},
/* 8*/	{'O', "", "--backup-outdate", ARG_DOUBLE, 0, 0, /* &outdate_time_float */ 0},
/* 9*/	{'o', "", "--only-read", ARG_SET, 0, 0, &only_read},
/*20*/	{'p', "", "--password", ARG_CALLBACK, 0, 0, (void *) opt_password_callback},
/* 1*/	{'R', "", "--exclude-regexp", ARG_CALLBACK, 0, 0, (void *) opt_exclude_callback},
/* 2*/	{'r', "", "--restore-access", ARG_SET, 0, 0, &rest_access},
/* 3*/	{'s', "", "--starting-file", ARG_STRING, &start_file, 0, 0},
/* 4*/	{'S', "", "--suffix", ARG_STRING, &backup_extension, 0, 0},
/* 5*/	{'T', "", "--allow-matches", ARG_STRING, /* &allow_str */ 0, 0, 0},
/* 6*/	{'t', "--dry-run", "--test-only", ARG_SET, 0, 0, &only_test},
/* 7*/	{'v', "", "--verbose", ARG_ADD, 0, 0, &verbose},
/* 8*/	{'V', "", "--version", ARG_SET, 0, 0, &show_version},
/* 9*/	{'X', "", "--exclude", ARG_CALLBACK, 0, 0, (void *) opt_exclude_callback},
#ifdef HAVE_GETTIMEOFDAY
	{0, "", "--nice", ARG_INT, 0, 0, &nice_to_cpu},
#endif
	{0, "", "--no-mtimes", ARG_SET, 0, 0, &no_mtimes},
	{0, "", "--no-chmod", ARG_CLEAR, 0, 0, &allow_chmod},
	{0, "", "--no-chown", ARG_CLEAR, 0, 0, &allow_chown},
	{0, "", "--no-netrc", ARG_CLEAR, 0, 0, &allow_netrc},
	{0, "", "--netrc", ARG_SET, 0, 0, &allow_netrc},
	{0, "", "--ignore-size", ARG_SET, 0, 0, &ignore_size},
	{0, "", "--allow-empty-ftp-dirs", ARG_SET, 0, 0, &ftpfs_option_allow_empty_directories},
      	{'L', "", "--strict-locking", ARG_CLEAR, 0, 0, &only_read},
        {'i', "", "--ignore-next-exclude", ARG_ADD, 0, 0, &ignore_next_exclude},
	{0, 0, 0, 0, 0, 0, 0}
    };
    arg_opts[10].str = &exclude_file;
    arg_opts[15].str = &max_str;
    arg_opts[18].option = &outdate_time_float;
    arg_opts[25].str = &allow_str;

    control_mirror_directories = malloc (argc * sizeof(char *));
    arg_opts[0].strs = control_mirror_directories;
    memset (control_mirror_directories, 0, argc * sizeof(char *));

#ifdef HAVE_GETCWD
    getcwd (cwd, MAX_PATH_LEN);
#else
    getwd (cwd);
#endif

    if (strcmp (progname, "mirrordir")) {
/* defaults for copydir */
	dont_remove = 1;
	copy_mode = 1;
	erase_directories = 0;
    }

    i = get_cmdline_options (argc, argv, arg_opts);

    if (show_help)
	usage (0);

    if (show_version) {
	fprintf (stdout, "Mirrordir version " VERSION "\n");
	exit (0);
    }

    if (i) {
	progmess ("error on cmdline near", argv[i]);
	return 1;
    }

    if (!control_mirror_directories[0] || !control_mirror_directories[1])
	return 1;
    if (copy_mode) {
	char **t;
	for (t = control_mirror_directories;*t;t++)
	    *t = pathdup (*t);
	t--;
	mirror_dir = *t;
	control_files = control_mirror_directories;
	*t = 0;
    } else {
	if (control_mirror_directories[2])
	    return 1;
	free_me[free_num++] = control_dir = pathdup (control_mirror_directories[0]);
	free_me[free_num++] = mirror_dir = pathdup (control_mirror_directories[1]);
    }

/* now apply some rules */
    if (copy_mode)
	dont_remove = 1;

    if (access_times)
	rest_access = 1;

    if (backup_extension)
	if (!backup_howmany)
	    backup_howmany = 1;

    if (backup_howmany)
	if (!backup_extension)
	    backup_extension = strdup (".OLD-%d");

    if (start_file)
	if (block_pos == BLOCK_NOT_USED)
	    block_pos = BLOCK_BEFORE_START;

    if (!block_size)
	block_size = 1024;

/* finally process some remaining args */
    outdate_time = (time_t) outdate_time_float;

    if (max_str)
	if (calc_bytes (max_str))
	    return 1;

    if (allow_str)
	if (parse_permissions_command (allow_str, &allow_permissions_mask))
	    return 1;

    if (exclude_file)
	if (load_exclude_file (exclude_file))
	    return 1;
    return 0;
}

/* }}} command line processing */

/* {{{ vfs callbacks */

void message_callback (char *p, ...)
{
    char s[1024];
    va_list pa;
    if (!verbose)
	return;
    va_start (pa, p);
    vsprintf (s, p, pa);
    va_end (pa);
    p = s;
    while ((p = strchr (p, '\n')))
	*p = ' ';
    if (strstr (s, "etting file") || strstr (s, "toring file")) {
	fprintf (stdout, "%s", s);
	fflush (stdout);
	fprintf (stdout, "\r");
    } else {
	verbmess (s, 0);
    }
}

char *anonymous_password_callback (char *host)
{
    char *p;

    /* password explicitly specified on commandline takes precedence */
    if (ftpfs_password)
	return ftpfs_password;
    /* then have a look at ~/.netrc */
    if (allow_netrc
	&& (p = lookup_netrc (host, "anonymous"))) {	/*EK */
	if (verbose)
	    verbmess ("found password for anonymous in ./netrc for", host);
	return p;
    }
    return NULL;
}

char *password_callback (char *a, char *b, char *c, char *host, char *user)
{				/*EK */
    char *p;

    /* password explicitly specified on commandline takes precedence */
    if (ftpfs_password)
	return ftpfs_password;
    /* then have a look at ~/.netrc */
    if (allow_netrc
	&& (p = lookup_netrc (host, user))) {	/*EK */
	if (verbose)
	    verbmess ("found password in ./netrc for", host);
	return p;
    }
    /* if everything else fails, then ask */
    return strdup (getpass (b));
}

void check_interrupt (void)
{
    if (terminate) {
	if (verbose) {
	    char s[12];
	    sprintf (s, "%d", terminate);
	    verbmess ("terminating on signal", s);
	}
	vfs_shut ();
	exit (1);
    }
}

int gotinterrupt_callback (void)
{
    int t;
    t = interupted;
    interupted = 0;
    return t;
}

/* }}} vfs callbacks */

/* {{{ signal handling */

#if (RETSIGTYPE==void)
#define handler_return return
#else
#define handler_return return 1
#endif

static RETSIGTYPE quit_handler (int x)
{
    interupted = 1;
    terminate = x;
    handler_return;
}

/* }}} signal handling */

/* {{{ main function */

int main (int argc, char **argv)
{
    char *p;
    int r;

    memset (free_me, 0, sizeof (void *) * 256);

    p = strrchr (argv[0], '/');
    progname = strdup (p ? p + 1 : argv[0]);

    signal (SIGQUIT, quit_handler);
    signal (SIGINT, quit_handler);
    signal (SIGTERM, quit_handler);

    if (pars_opts (argc, argv))
	usage (1);
    if (show_version) {
	fprintf (stdout, "Mirrordir version " VERSION "\n");
	exit (0);
    }
    if (copy_mode) {
	char **n;
	for (n = control_files; *n; n++) {
	    if (!strcmp (mirror_dir, *n)) {
		progmess ("copying a path to the same path", *n);
		usage (1);
	    }
	}
    } else {
	if (!strcmp (mirror_dir, control_dir)) {
	    progmess ("mirror and control directories are the same path", control_dir);
	    usage (1);
	}
    }
    vfs_set_message_callback (message_callback);
    vfs_set_getpasswd_callback (password_callback);
    vfs_set_getanonpasswd_callback (anonymous_password_callback);
    vfs_set_gotinterrupt_callback (gotinterrupt_callback);

    vfs_init ();
    ftpfs_init_passwd ();

    if (copy_mode) {
	r = recurs_dir (0, mirror_dir, control_files);
    } else {
	r = recurs_dir (control_dir, mirror_dir);
    }
    vfs_shut ();

    if (verbose && handle_hardlinks)
	print_hardlinks ();

    if (verbose) {
	char s[16];
	sprintf (s, "%lukB", blocks_to_kilobytes (blocks_written));
	infomess ("Total mirrored", s);
    }
    free (progname);
    if (control_mirror_directories)
	free (control_mirror_directories);
    while (free_num--)
	if (free_me[free_num])
	    free (free_me[free_num]);
    free_all_hardlinks ();

    mad_finalize (__FILE__, __LINE__);
/* FIXME: should other return values be reported. Most people won't now how to read them anyway */
    if (r == -1 || real_bad_error) {
	if (file_in_use == real_bad_error)
	    return RETURN_VALUE_FILE_IN_USE;
	return RETURN_VALUE_ERROR;
    }
    return RETURN_VALUE_SUCCESS;
}

/* }}} main function */


