/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: init.c,v 6.1.1.1 97/02/24 14:52:50 nevin Exp $
 *
 *	Function:	- initialize the MPI session
 *			- the kitchen-sink of MPI
 *	Accepts:	- ptr to argc
 *			- ptr to argv
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>
#include <string.h>

#include <all_list.h>
#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <net.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * static variables
 */
static int		maxtag;			/* max. tag attribute */
static int		host;			/* host rank attribute */
static int		io;			/* I/O rank attribute */
static int		wtimeglob;		/* global time attribute */
static int		mpi_nprocs;		/* # world processes */
static int		mpi_nparent;		/* # parent processes */
static int		mpi_cid;		/* CID of parent intercomm */
static struct _gps	*mpi_procs;		/* array world & parent GPS */

/*
 * external functions
 */
extern struct _proc *	lam_procfind();
extern void		lam_resetfunc();
extern void		lam_setcid();
extern void		lam_setfunc();
extern int		lam_errfunc();
extern int		lam_linit();
extern int		lam_mkerr();
extern int		lam_tr_comm();
extern int		lam_tr_init();
extern int		lam_tr_on();

/*
 * local functions
 */
static int		init_comm();
static int		init_env();
static int		init_rdtype();
static void		init_dtype();
static void		init_errhdl();
static void		init_op();
static void		make_dtype();
static void		make_op();
static int		comm_cmp();
static int		port_cmp();

/*
 * local structures for maxloc/minloc operations
 */
struct flt_int {
	float		fi_float;
	int		fi_int;
};

struct dbl_int {
	double		di_double;
	int		di_int;
};

struct longdbl_int {
#if __STDC__
	long double	ldi_double;
#else	
	double		ldi_double;
#endif
	int		ldi_int;
};

struct long_int {
	long		li_long;
	int		li_int;
};

struct short_int {
	short		li_short;
	int		li_int;
};

/*
 * Create the whole MPI universe.
 */
int
MPI_Init(pargc, pargv)

int			*pargc;
char			***pargv;

{
	int		fl_init;		/* already init'ed? */
	int		err;			/* return error code */
	int		root;			/* root in parent comm if any */
	char		*name;			/* program name */

	lam_setfunc(BLKMPIINIT);
/*
 * Check if we have been initialized or finalized.
 */
	MPI_Initialized(&fl_init);
	if (fl_init || lam_finalized()) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINIT,
				lam_mkerr(MPI_ERR_OTHER, EMPIINIT)));
	}
/*
 * Initialize LAM.
 */
	name = (pargc && pargv && (*pargc > 0)) ? **pargv : 0;

	if (lam_linit(name, &mpi_nprocs, &mpi_nparent, &mpi_cid,
			&mpi_procs, &root)) {
		terror("MPI_Init: LAM error");
		exit(errno);
	}
/*
 * Initialize MPI pre-defined "stuff".
 * The order below is important, so don't "hack".
 */
	init_errhdl();
	init_dtype();
	init_op();

	if (init_comm()) {
		free((char *) mpi_procs);
		return(lam_errfunc(MPI_COMM_NULL,
			BLKMPIINIT, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	if (init_rdtype() || init_env()) {
		return(lam_errfunc(MPI_COMM_NULL,
			BLKMPIINIT, lam_mkerr(MPI_ERR_OTHER, errno)));
	}
/*
 * Initialize port list.
 */
	lam_ports = al_init(sizeof(struct _port), port_cmp);
	if (lam_ports == 0) return(LAMERROR);
/*
 * Initialize profiling package.
 */
	MPI_Pcontrol(1);
/*
 * cleanup
 */
	free((char *) mpi_procs);
/*
 * Make sure everyone else has also initialized.
 * Rank 0 of a spawned world must inform the root parent when this is so.
 */
	err = MPI_Barrier(MPI_COMM_WORLD);
	if (err != MPI_SUCCESS) return(LAMERROR);

	if ((mpi_nparent > 0) && (MPI_COMM_WORLD->c_group->g_myrank == 0)) {
		err = MPI_Send((void *) 0, 0, MPI_BYTE,
					root, 0, MPI_COMM_PARENT);
		if (err != MPI_SUCCESS) return(LAMERROR);
	}
/*
 * Record an initialization trace.
 * If we are tracing and trace generation is initially to be on, turn it on.
 */
	if (lam_tr_init(name, lam_clockskew)) return(LAMERROR);

	if ((_kio.ki_rtf & RTF_TRON) == RTF_TRON) {
		_kio.ki_rtf &= ~RTF_TRSWITCH;
		if (lam_tr_on() < 0) return(LAMERROR);
	}

	lam_resetfunc(BLKMPIINIT);
	return(MPI_SUCCESS);
}

/*
 *	init_errhdl
 *
 *	Function:	- initialize pre-defined error handles
 */
static void
init_errhdl()

{
	MPI_ERRORS_ARE_FATAL->eh_func = lam_errfatal;
	MPI_ERRORS_ARE_FATAL->eh_f77hdl = -1;
	MPI_ERRORS_ARE_FATAL->eh_refcount = 1;

	MPI_ERRORS_RETURN->eh_func = lam_errreturn;
	MPI_ERRORS_RETURN->eh_f77hdl = -1;
	MPI_ERRORS_RETURN->eh_refcount = 1;
}

/*
 *	init_comm
 *
 *	Function:	- initialize pre-defined communicators
 *	Returns:	- 0 or LAMERROR
 */
static int
init_comm()

{
	MPI_Group	group;			/* a process group */
	MPI_Comm	comm;			/* a communicator */
	int		i;
	struct _gps	*g;
	struct _proc	**p;
/*
 * Create the empty group.
 */
	MPI_GROUP_EMPTY->g_nprocs = 0;
	MPI_GROUP_EMPTY->g_myrank = MPI_UNDEFINED;
	MPI_GROUP_EMPTY->g_refcount = 1;
	MPI_GROUP_EMPTY->g_procs = 0;
/*
 * Initialize the list of communicators.
 */
	lam_comms = al_init(sizeof(MPI_Comm), comm_cmp);
	if (lam_comms == 0) return(LAMERROR);
/*
 * Create the "world" communicator.
 */
	comm = MPI_COMM_WORLD;

	group = (MPI_Group) malloc((unsigned) sizeof(struct _group) +
					(mpi_nprocs * sizeof(struct _proc *)));
	if (group == 0) return(LAMERROR);

	group->g_nprocs = mpi_nprocs;
	group->g_myrank = MPI_UNDEFINED;
	group->g_refcount = 1;
	group->g_procs = (struct _proc **)
				((char *) group + sizeof(struct _group));
/*
 * Fill the group processes and determine my rank.
 */
	g = mpi_procs;
	p = group->g_procs;

	for (i = 0; i < mpi_nprocs; ++i, ++g, ++p) {

		if ((*p = lam_procfind(g)) == 0) {
			return(LAMERROR);
		}
		if (*p == lam_myproc) {
			group->g_myrank = i;
		}
		(*p)->p_refcount++;
	}

	if (lam_comm_new(0, group, MPI_GROUP_NULL, LAM_PREDEF, &comm)) {
		return(LAMERROR);
	}

	if (!al_insert(lam_comms, &comm)) return(LAMERROR);
/*
 * Set the pid under which run time traces are stored and write the
 * trace for MPI_COMM_WORLD.
 */
	_kio.ki_tid = lam_myproc->p_gps.gps_pid;
	if (lam_tr_comm(MPI_COMM_WORLD)) return(LAMERROR);
/*
 * Create the "self" communicator.
 */
	comm = MPI_COMM_SELF;

	group = (MPI_Group) malloc((unsigned) sizeof(struct _group) +
						sizeof(struct _proc *));
	if (group == 0) return(LAMERROR);

	group->g_nprocs = 1;
	group->g_myrank = 0;
	group->g_refcount = 1;
	group->g_procs = (struct _proc **)
				((char *) group + sizeof(struct _group));

	group->g_procs[0] = lam_myproc;
	lam_myproc->p_refcount++;
	
	if (lam_comm_new(1, group, MPI_GROUP_NULL, LAM_PREDEF, &comm)) {
		return(LAMERROR);
	}

	if (!al_insert(lam_comms, &comm)) return(LAMERROR);
/*
 * Create the parent intercommunicator.
 */
	comm = MPI_COMM_PARENT;
	
	if (mpi_nparent > 0) {

		group = (MPI_Group) malloc((unsigned) sizeof(struct _group) +
					(mpi_nparent * sizeof(struct _proc *)));
		if (group == 0) return(LAMERROR);

		group->g_nprocs = mpi_nparent;
		group->g_myrank = MPI_UNDEFINED;
		group->g_refcount = 1;
		group->g_procs = (struct _proc **)
				((char *) group + sizeof(struct _group));
/*
 * Fill the parent group processes.
 */
		g = mpi_procs + mpi_nprocs;
		p = group->g_procs;

		for (i = 0; i < mpi_nparent; ++i, ++g, ++p) {
			*p = lam_procfind(g);
			if (*p == 0) return(LAMERROR);
			(*p)->p_refcount++;
		}
		
		if (lam_comm_new(mpi_cid, MPI_COMM_WORLD->c_group,
				group, LAM_CINTER | LAM_PREDEF, &comm)) {
			return(LAMERROR);
		}
	} else {
		if (lam_comm_new(2, MPI_COMM_WORLD->c_group, MPI_GROUP_EMPTY,
				LAM_CINTER | LAM_PREDEF, &comm)) {
			return(LAMERROR);
		}
	}
	MPI_COMM_WORLD->c_group->g_refcount++;

	if (!al_insert(lam_comms, &comm)) return(LAMERROR);
	if (lam_tr_comm(MPI_COMM_PARENT)) return(LAMERROR);
	lam_setcid(MPI_COMM_PARENT->c_contextid);

	return(0);
}

/*
 *	init_dtype
 *
 *	Function:	- initialize basic (intrinsic) datatypes
 */
static void
init_dtype()

{
/*
 * common datatypes
 */
	make_dtype(MPI_BYTE, sizeof(char), sizeof(char), TRDTBYTE);
	make_dtype(MPI_PACKED, sizeof(char), sizeof(char), TRDTPACKED);
	make_dtype(MPI_UB, 0, 1, TRDTUB);
	make_dtype(MPI_LB, 0, 1, TRDTLB);
/*
 * C datatypes
 */
	make_dtype(MPI_CHAR, sizeof(char), sizeof(char), TRDTCHAR);
	make_dtype(MPI_SHORT, sizeof(short), ALIGNMENT_SHORT, TRDTSHORT);
	make_dtype(MPI_INT, sizeof(int), ALIGNMENT_INT, TRDTINT);
	make_dtype(MPI_LONG, sizeof(long), ALIGNMENT_LONG, TRDTLONG);
	make_dtype(MPI_FLOAT, sizeof(float), ALIGNMENT_FLOAT, TRDTFLOAT);
	make_dtype(MPI_DOUBLE, sizeof(double), ALIGNMENT_DOUBLE, TRDTDOUBLE);
#if __STDC__
	make_dtype(MPI_LONG_DOUBLE,
		sizeof(long double), ALIGNMENT_LONG_DOUBLE, TRDTLONGDOUBLE);
#else
	make_dtype(MPI_LONG_DOUBLE,
		sizeof(double), ALIGNMENT_DOUBLE, TRDTLONGDOUBLE);
#endif
	make_dtype(MPI_UNSIGNED_CHAR,
		sizeof(unsigned char), sizeof(unsigned char), TRDTUCHAR);
	make_dtype(MPI_UNSIGNED_SHORT,
		sizeof(unsigned short), ALIGNMENT_SHORT, TRDTUSHORT);
	make_dtype(MPI_UNSIGNED,
		sizeof(unsigned int), ALIGNMENT_INT, TRDTUINT);
	make_dtype(MPI_UNSIGNED_LONG,
		sizeof(unsigned long), ALIGNMENT_LONG, TRDTULONG);
/*
 * FORTRAN datatypes
 */
	make_dtype(MPI_F_CHARACTER, sizeof(char), sizeof(char), TRDTFCHARACTER);
	make_dtype(MPI_F_LOGICAL, sizeof(int), ALIGNMENT_INT, TRDTFLOGICAL);
	make_dtype(MPI_F_INTEGER, sizeof(int), ALIGNMENT_INT, TRDTFINTEGER);
	make_dtype(MPI_F_REAL, sizeof(float), ALIGNMENT_FLOAT, TRDTFREAL);
	make_dtype(MPI_F_DOUBLE_PRECISION, 
		sizeof(double), ALIGNMENT_DOUBLE, TRDTFDBLPREC);
	make_dtype(MPI_F_COMPLEX,
		2 * sizeof(float), ALIGNMENT_FLOAT, TRDTFCOMPLEX);
	make_dtype(MPI_F_DOUBLE_COMPLEX,
		2 * sizeof(double), ALIGNMENT_DOUBLE, TRDTFDBLCOMPLEX);
}

/*
 *	init_rdtype
 *
 *	Function:	- initialize the reduction datatypes
 *	Returns:	- 0 or LAMERROR
 */
static int
init_rdtype()

{
	MPI_Datatype	new;			/* new datatype */
	MPI_Datatype	types[2];		/* struct datatypes */
	int		lengths[2];		/* struct lengths */
	MPI_Aint	disps[2];		/* struct displacements */
	struct flt_int	fi[2];			/* float_int data */
	struct dbl_int	di[2];			/* double_int data */
	struct long_int	li[2];			/* long_int data */
	struct short_int
			si[2];			/* short_int data */
	struct longdbl_int
			ldi[2];			/* longdbl_int data */
	int		err;			/* error code */
/*
 * Create MPI_2INT.
 */
	err = MPI_Type_contiguous(2, MPI_INT, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_2INT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_2INT->dt_label = TRDT2INT;
	MPI_2INT->dt_commit = 1;
	MPI_2INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_2FLOAT.
 */
	err = MPI_Type_contiguous(2, MPI_FLOAT, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_2FLOAT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_2FLOAT->dt_label = TRDT2FLOAT;
	MPI_2FLOAT->dt_commit = 1;
	MPI_2FLOAT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_2DOUBLE.
 */
	err = MPI_Type_contiguous(2, MPI_DOUBLE, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_2DOUBLE, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_2DOUBLE->dt_label = TRDT2DOUBLE;
	MPI_2DOUBLE->dt_commit = 1;
	MPI_2DOUBLE->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_FLOAT_INT.
 */
	types[0] = MPI_FLOAT; types[1] = MPI_INT;
	lengths[0] = 1; lengths[1] = 1;

	MPI_Address(&fi[0], &disps[0]);
	MPI_Address(&(fi[0].fi_int), &disps[1]);

	disps[1] -= disps[0]; disps[0] = 0;

	err = MPI_Type_struct(2, lengths, disps, types, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_FLOAT_INT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_FLOAT_INT->dt_label = TRDTFLOATINT;
	MPI_FLOAT_INT->dt_commit = 1;
	MPI_FLOAT_INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_DOUBLE_INT.
 */
	types[0] = MPI_DOUBLE; types[1] = MPI_INT; 
	lengths[0] = 1; lengths[1] = 1;

	MPI_Address(&di[0], &disps[0]);
	MPI_Address(&(di[0].di_int), &disps[1]);

	disps[1] -= disps[0]; disps[0] = 0;

	err = MPI_Type_struct(2, lengths, disps, types, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_DOUBLE_INT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_DOUBLE_INT->dt_label = TRDTDOUBLEINT;
	MPI_DOUBLE_INT->dt_commit = 1;
	MPI_DOUBLE_INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_LONG_DOUBLE_INT.
 */
	types[0] = MPI_LONG_DOUBLE; types[1] = MPI_INT;
	lengths[0] = 1; lengths[1] = 1;

	MPI_Address(&ldi[0], &disps[0]);
	MPI_Address(&(ldi[0].ldi_int), &disps[1]);

	disps[1] -= disps[0]; disps[0] = 0;

	err = MPI_Type_struct(2, lengths, disps, types, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_LONG_DOUBLE_INT,
		(char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_LONG_DOUBLE_INT->dt_label = TRDTLONGDBLINT;
	MPI_LONG_DOUBLE_INT->dt_commit = 1;
	MPI_LONG_DOUBLE_INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_LONG_INT.
 */
	types[0] = MPI_LONG; types[1] = MPI_INT;
	lengths[0] = 1; lengths[1] = 1;

	MPI_Address(&li[0], &disps[0]);
	MPI_Address(&(li[0].li_int), &disps[1]);

	disps[1] -= disps[0]; disps[0] = 0;

	err = MPI_Type_struct(2, lengths, disps, types, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_LONG_INT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_LONG_INT->dt_label = TRDTLONGINT;
	MPI_LONG_INT->dt_commit = 1;
	MPI_LONG_INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_SHORT_INT.
 */
	types[0] = MPI_SHORT; types[1] = MPI_INT;
	lengths[0] = 1; lengths[1] = 1;

	MPI_Address(&si[0], &disps[0]);
	MPI_Address(&(si[0].li_int), &disps[1]);

	disps[1] -= disps[0]; disps[0] = 0;

	err = MPI_Type_struct(2, lengths, disps, types, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_SHORT_INT, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_SHORT_INT->dt_label = TRDTSHORTINT;
	MPI_SHORT_INT->dt_commit = 1;
	MPI_SHORT_INT->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_2INTEGER.
 */
	err = MPI_Type_contiguous(2, MPI_F_INTEGER, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_F_2INTEGER, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_F_2INTEGER->dt_label = TRDTF2INTEGER;
	MPI_F_2INTEGER->dt_commit = 1;
	MPI_F_2INTEGER->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_2REAL.
 */
	err = MPI_Type_contiguous(2, MPI_F_REAL, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_F_2REAL, (char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_F_2REAL->dt_label = TRDTF2REAL;
	MPI_F_2REAL->dt_commit = 1;
	MPI_F_2REAL->dt_flags |= LAM_PREDEF;
/*
 * Create MPI_2DOUBLE_PRECISION.
 */
	err = MPI_Type_contiguous(2, MPI_F_DOUBLE_PRECISION, &new);
	if (err != MPI_SUCCESS) return(LAMERROR);

	memcpy((char *) MPI_F_2DOUBLE_PRECISION,
			(char *) new, sizeof(struct _dtype));
	free((char *) new);
	MPI_F_2DOUBLE_PRECISION->dt_label = TRDTF2DBLPREC;
	MPI_F_2DOUBLE_PRECISION->dt_commit = 1;
	MPI_F_2DOUBLE_PRECISION->dt_flags |= LAM_PREDEF;

	return(0);
}

/*
 *	init_op
 *
 *	Function:	- initialize intrinsic reduction operations
 */
static void
init_op()

{
	make_op(MPI_MAX, lam_max);
	make_op(MPI_MIN, lam_min);
	make_op(MPI_SUM, lam_sum);
	make_op(MPI_PROD, lam_prod);
	make_op(MPI_LAND, lam_land);
	make_op(MPI_BAND, lam_band);
	make_op(MPI_LOR, lam_lor);
	make_op(MPI_BOR, lam_bor);
	make_op(MPI_LXOR, lam_lxor);
	make_op(MPI_BXOR, lam_bxor);
	make_op(MPI_MAXLOC, lam_maxloc);
	make_op(MPI_MINLOC, lam_minloc);
}

/*
 *	init_env
 *
 *	Function:	- initialize environment attributes
 *	Returns:	- 0 or LAMERROR
 */
static int
init_env()

{
	MPI_Group	world;			/* world group */
	struct _attrkey	*keystate;		/* key state */
	struct _proc	**p;			/* process */
	int		key;			/* attribute key */
	int		err;			/* error code */
	int		i;
/*
 * Create the predefined keys.
 */
	err = MPI_Keyval_create(MPI_NULL_COPY_FN, MPI_NULL_DELETE_FN,
				&key, (void *) 0);
	if ((err != MPI_SUCCESS) || (key != MPI_TAG_UB)) return(LAMERROR);

	err = MPI_Keyval_create(MPI_NULL_COPY_FN, MPI_NULL_DELETE_FN,
				&key, (void *) 0);
	if ((err != MPI_SUCCESS) || (key != MPI_HOST)) return(LAMERROR);

	err = MPI_Keyval_create(MPI_NULL_COPY_FN, MPI_NULL_DELETE_FN,
				&key, (void *) 0);
	if ((err != MPI_SUCCESS) || (key != MPI_IO)) return(LAMERROR);

	err = MPI_Keyval_create(MPI_NULL_COPY_FN, MPI_NULL_DELETE_FN,
				&key, (void *) 0);
	if ((err != MPI_SUCCESS) || (key != MPI_WTIME_IS_GLOBAL)) {
		return(LAMERROR);
	}

	err = MPI_Keyval_create(MPI_NULL_COPY_FN, MPI_NULL_DELETE_FN,
				&key, (void *) 0);
	if ((err != MPI_SUCCESS) || (key != MPI_UNIVERSE_SIZE)) {
		return(LAMERROR);
	}
/*
 * Initialize the attribute values.
 */
	maxtag = LAM_MAXTAG;
	wtimeglob = 0;
/*
 * We interpret MPI_IO to include the ability to print to a terminal.
 * MPI_IO is set to my rank if I'm on the origin node.
 * Otherwise, it's set to MPI_ANY_SOURCE if all processes are on
 * the origin node, MPI_PROC_NULL if no processes are on the
 * origin node or the lowest ranked process on the origin node.
 */
	world = MPI_COMM_WORLD->c_group;
	p = world->g_procs;

	for (i = 0; i < world->g_nprocs; ++i) {
		if ((*p)->p_gps.gps_node != getorigin()) break;
		p++;
	}

	if (i >= world->g_nprocs) {
		io = MPI_ANY_SOURCE;
	} else if (lam_myproc->p_gps.gps_node == getorigin()) {
		io = world->g_myrank;
	} else {
		p = world->g_procs;

		for (i = 0; i < world->g_nprocs; ++i) {
			if ((*p)->p_gps.gps_node == getorigin()) break;
			p++;
		}

		io = (i >= world->g_nprocs) ? MPI_PROC_NULL : i;
	}
/*
 * MPI_HOST is similarly related to the origin node, except if all
 * processes are on the origin, we set this attribute to rank 0.
 */
	p = world->g_procs;

	for (i = 0; i < world->g_nprocs; ++i) {
		if ((*p)->p_gps.gps_node == getorigin()) break;
		p++;
	}

	host = (i >= world->g_nprocs) ? MPI_PROC_NULL : i;
/*
 * The universe size is inherited from the parents or is the number of
 * nodes in the LAM.
 */
	if (lam_universe_size < 0) {
		if ((lam_universe_size = getnall()) < 0) {
			return(LAMERROR);
		}
	}
/*
 * Store the attribute values.
 */
	err = MPI_Attr_put(MPI_COMM_WORLD, MPI_TAG_UB,
		((lam_f77init) ? (void *) maxtag : (void *) &maxtag));
	if (err != MPI_SUCCESS) return(LAMERROR);

	err = MPI_Attr_put(MPI_COMM_WORLD, MPI_HOST,
		((lam_f77init) ? (void *) host : (void *) &host));
	if (err != MPI_SUCCESS) return(LAMERROR);

	err = MPI_Attr_put(MPI_COMM_WORLD, MPI_IO,
		((lam_f77init) ? (void *) io : (void *) &io));
	if (err != MPI_SUCCESS) return(LAMERROR);

	err = MPI_Attr_put(MPI_COMM_WORLD, MPI_WTIME_IS_GLOBAL,
		((lam_f77init) ? (void *) wtimeglob : (void *) &wtimeglob));
	if (err != MPI_SUCCESS) return(LAMERROR);
	
	err = MPI_Attr_put(MPI_COMM_WORLD, MPI_UNIVERSE_SIZE,
		((lam_f77init) ? (void *) lam_universe_size :
			(void *) &lam_universe_size));
	if (err != MPI_SUCCESS) return(LAMERROR);
/*
 * Mark them as predefined.
 */
	if ((keystate = lam_getattr(MPI_TAG_UB)) == 0) return(LAMERROR);
	keystate->ak_flags = LAM_PREDEF;
	
	if ((keystate = lam_getattr(MPI_HOST)) == 0) return(LAMERROR);
	keystate->ak_flags = LAM_PREDEF;

	if ((keystate = lam_getattr(MPI_IO)) == 0) return(LAMERROR);
	keystate->ak_flags = LAM_PREDEF;

	if ((keystate = lam_getattr(MPI_WTIME_IS_GLOBAL)) == 0) {
		return(LAMERROR);
	}
	keystate->ak_flags = LAM_PREDEF;

	if ((keystate = lam_getattr(MPI_UNIVERSE_SIZE)) == 0) {
		return(LAMERROR);
	}
	keystate->ak_flags = LAM_PREDEF;

	return(0);
}

/*
 *	make_dtype
 *
 *	Function:	- create a basic datatype
 *			- no errors happen here
 *	Accepts:	- MPI datatype
 *			- size of datatype
 *			- datatype label
 */
static void
make_dtype(dtype, size, align, label)

MPI_Datatype		dtype;
int			size;
int			align;
int			label;

{
	dtype->dt_format = LAM_DTBASIC;
	dtype->dt_flags = LAM_DTNOPACK | LAM_DTNOXADJ | LAM_PREDEF;
	dtype->dt_commit = 1;
	dtype->dt_label = label;
	dtype->dt_refcount = 1;
	dtype->dt_align = align;
	dtype->dt_upper = dtype->dt_dataup = size;
	dtype->dt_lower = dtype->dt_datalow = 0;
	dtype->dt_size = size;
	dtype->dt_nelem = 1;
	dtype->dt_count = 0;
	dtype->dt_length = 0;
	dtype->dt_stride = 0;
	dtype->dt_dtype = 0;
	dtype->dt_lengths = 0;
	dtype->dt_disps = 0;
	dtype->dt_dtypes = 0;

	if (dtype == MPI_UB) {
		dtype->dt_flags |= LAM_DTHASUB;
		dtype->dt_nelem = 0;
	}
	else if (dtype == MPI_LB) {
		dtype->dt_flags |= LAM_DTHASLB;
		dtype->dt_nelem = 0;
	}
}

/*
 *	make_op
 *
 *	Function:	- create intrinsic reduction operation
 *			- no errors happen here
 *	Accepts:	- MPI reduction operation
 *			- reduction function
 */
static void
make_op(op, func)

MPI_Op			op;
void			(*func)();

{
	op->op_func = func;
	op->op_commute = 1;
	op->op_f77dtype = -1;
	op->op_flags = LAM_PREDEF;
}

/*
 *	comm_cmp
 *
 *	Function:	- compare two communicators
 *	Accepts:	- ptr to two entries
 *	Returns:	- 0 if same communicator, else 1
 */
static int
comm_cmp(c1, c2)

MPI_Comm		*c1, *c2;

{
	return( !( ((*c1)->c_contextid == (*c2)->c_contextid)
			|| ((*c1)->c_contextid == -((*c2)->c_contextid + 1))));
}

/*
 *	port_cmp
 *
 *	Function:	- compare two ports
 *	Accepts:	- ptr to two entries
 *	Returns:	- string comparison of the port names
 */
static int
port_cmp(p1, p2)

struct _port		*p1, *p2;

{
	return(strcmp(p1->prt_name, p2->prt_name));
}
