/*
  copyright   : GPL
  title       : fakeroot
  description : create a "fake" root shell, by wrapping 
                functions like chown, stat, etc. Usefull for debian
                packaging mechanism
  author      : joost witteveen, joostje@debian.org
*/

/*
  upon startup, fakeroot creates three instances of itself
  (forks twice):
    1) parent, creates ipc message queues, creates children,
       waits for children to die, and then clears the ipc message
       queues.

    2) the ``main'' daemon, recieves ipc messages, maintains
       fake inode<->ownership database (actually just a 
       lot of struct stat entries).

    3) one instance that execve's the command to run.
       (with LD_PRELOAD=libfakeroot, and FAKEROOTKEY=ipc-key,
       thus the executed commands will communicate with
       instance 2).

  The following functions are currently wrapped:
     getuid(), geteuid(), getgid(), getegid(),
     mknod()
     chown(), fchown()
     chmow(), fchmod()
     mkdir(),
     lstat(), fstat(), stat() (actually, __xlstat, ...)
     unlink(), remove(), rmdir(), rename()
    
  comments:
    I need to wrap unlink because of the following:
        install -o admin foo bar
	rm bar
        touch bar         //bar now may have the same inode:dev as old bar,
	                  //but unless the rm was caught,
			  //fakeroot still has the old entry.
        ls -al bar
    Same goes for all other ways to remove inodes form the filesystem,
    also rename(existing_file, any_file).
    */			       
/* ipc documentation bugs: msgsnd(2): MSGMAX=4056, not 4080 
   (def in ./linux/msg.h, couldn't find other def in /usr/include/ *  */


#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/wait.h>
#include <sys/sem.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "fakeroot.h"
#include <set>
#include <string.h>

#include <iostream.h>

char *fakelib=NULL;
bool   mixed_libc_hack=true;
bool   debian_pedantic=false;

void process_chown(struct my_msgbuf *buf);
void process_fchown(struct my_msgbuf *buf);
void process_chmod(struct my_msgbuf *buf);
void process_fchmod(struct my_msgbuf *buf);
void process_mknod(struct my_msgbuf *buf);
void process_stat(struct my_msgbuf *buf);
//void process_fstat(struct my_msgbuf *buf);
void process_unlink(struct my_msgbuf *buf);
void process_checklibc(struct my_msgbuf *buf);
void process_debianpedantic(struct my_msgbuf *buf);
void process_debug(struct my_msgbuf *buf);
void process_debugdata(struct my_msgbuf *buf);
bool operator <(const struct stat &a, const struct stat &b);

typedef void (*process_func)(struct my_msgbuf *);

process_func func_arr[]={process_chown,process_fchown,
			 process_chmod,process_fchmod,
			 process_mknod,
			 process_stat,//process_fstat,
			 process_unlink,
			 process_checklibc,
			 process_debianpedantic,
                         process_debug,
			 process_debugdata};


key_t msg_key;
int msg_get,msg_send, sem_id;
;
bool debug=false;


// our main (and only) data base, contains all inodes and other info:
set <struct stat, less<struct stat> > data;

bool operator <(const struct stat &a, const struct stat &b){
  
  if(a.st_ino<b.st_ino)
    return true;
  else if(a.st_ino> b.st_ino)
    return false;
  else 
    return a.st_dev<b.st_dev;
  
  /*
    return a.st_ino<b.st_ino;
  */
};

/*********************************/
/*                               */
/* semun is no longer defined in */
/* glibc. if so, define it here. */
/*                               */
/*********************************/

#ifdef _SEM_SEMUN_UNDEFINED
static
  union semun
    {
      int val;
      struct semid_ds *buf;
      unsigned short int *array;
      struct seminfo *__buf;
    };
#endif


/*********************************/
/*                               */
/* hackery for mixed libc's      */
/*                               */
/*********************************/

int check_libc(const char *path){
  /* return 0 if path uses same libc as we do, 1 if different,
     and -1 if unsucessfull (file doesn't exist) */
  FILE *p;
  char buf[MAXPATH],script[MAXPATH];
  char *t;
  
  if(debug)
    fprintf(stderr,"fakeroot: checking libc version of %s\n",path);
  /* first check if path is a script; if so, try the binary the script uses*/
  while (1){
    p=fopen(path,"r");
    if(!p)
      return -1; /* OK, file _could_ be mode 111, but I don't care */
    fgets(buf,sizeof(buf),p);
    fclose(p);
    if(strncmp(buf,"#!",strlen("#!")))
      break;
    t=buf+strlen("#!");
    while(*t&&isspace(*t))
      t++;
    path=t;
    while(*t&&!isspace(*t))
      t++;
    *t=0;
    strcpy(script,path);
    path=script;
    if(debug)
      fprintf(stderr,"fakeroot: oops, a script. It's path is \"%s\"\n",path);
  }

  strcpy(buf,LDD_BINARY " ");
  strcat(buf,path);
  if(debug)
    fprintf(stderr,"fakeroot: ldd-ing: \"%s\"\n",buf);
  p=popen(buf,"ro");
  if(p==NULL){
    if(debug)
      fprintf(stderr,"fakeroot: ldd-ing: generated no pipe\n");
    return -1;
  }
  while(fgets(buf,sizeof(buf),p)){
    t=buf;
    if(debug)
      fprintf(stderr,"fakeroot: ldd-ing just read %s\n",buf);
    while(*t&&isspace(*t))
      t++;
    if(!strncmp(t,"libc.so.6",strlen("libc.so.6"))){
      while(fgets(buf,sizeof(buf),p));
      pclose(p);
      if(debug)
	fprintf(stderr,"fakeroot: Good, it's a libc6 executable\n");
      return 0;
    }
    if(!strncmp(t,"libc.so.5",strlen("libc.so.5"))){
      while(fgets(buf,sizeof(buf),p));
      pclose(p);
      if(debug)
	fprintf(stderr,"fakeroot: Sorry, it's a libc5 executable\n");
      return 1;
    }
  }
  pclose(p);
  return 0;
}


/*********************************/
/*                               */
/* basic communications          */
/*                               */
/*********************************/


void send_msg_size(struct my_msgbuf *buf,int size){
  int r;
  
  buf->mtype=1;
  r=msgsnd(msg_send, buf,
	   size+sizeof(struct q_and_a_func_id),
	   0);
  if(r)
    perror("FAKEROOT, when sending message");	 
}

void send_stat(struct my_msgbuf *buf){
  int r,l;

  l=sizeof(struct stat_param)+sizeof(struct q_and_a_func_id);

  buf->mtype=1;
  r=msgsnd(msg_send,buf, l, 0);
  if(r)
    perror("FAKEROOT, when sending message");	 
}


/*********************************/
/*                               */
/* data base maintainance        */
/*                               */
/*********************************/
void debug_stat(const struct stat *st){
  fprintf(stderr,"d:i=(%Li", st->st_dev);
  fprintf(stderr, ":%li))",st->st_ino);
  fprintf(stderr,", mode=0%o, nlink=%i, own=(%i,%i), size=%li\n",
	  st->st_mode,
	  st->st_nlink,
	  st->st_uid,st->st_gid,
	  st->st_size);
	 
  /*cerr<<"d:i=("<<st->st_dev<<":"<<st->st_ino
      <<"), mode="<<st->st_mode<<", nlink="<<st->st_nlink
      <<", own=("<<st->st_uid<<", "<<st->st_gid<<")"
      <<", size="<<st->st_size
      <<endl;
      */
}

void insert_or_overwrite(set <struct stat, less<struct stat> > &data,
			 struct stat &st){
  // just doing a data.insert(stat) would not update data
  // of inodes that are already present (if operator compares "equal",
  // stat isn't writen).

  set <struct stat, less<struct stat> >::iterator i;

  
  i=data.find(st);
  if(i==data.end()){
    if(debug){
      fprintf(stderr,"FAKEROOT: insert_or_overwrite unknown stat:\n");
      debug_stat(&st);
    }
    data.insert(st);
  }
  else
    memcpy(((struct stat *)&(*i)),&st,sizeof(st));
}


void copy_stat(struct stat *dest, const struct stat *source){
  //copy only the things we fake (i.e. "know better"), and leave
  //the other info (like file size etc).
    dest->st_dev =source->st_dev;
    dest->st_rdev=source->st_rdev;
    dest->st_ino =source->st_ino;
    dest->st_uid =source->st_uid;
    dest->st_gid =source->st_gid;
    dest->st_mode=source->st_mode;
}
int fake_stat(const char *path,
	      struct stat *st){
  int r;
  set <struct stat, less<struct stat> >::iterator i;

  r=lstat(path,st);
  if(r==-1)
    return r;
  i=data.find(*st);
  if(i!=data.end()){
    if(debug)
      fprintf(stderr,"FAKEROOT: stat known ");
    copy_stat(st,&(*i));
  }else{
    if(debug)
      fprintf(stderr,"FAKEROOT: stat unknown ");
    st->st_uid=0;
    st->st_gid=0;
  }
  if(debug){
    fprintf(stderr,"\"%s\" = ",path);
    debug_stat(st);
  }
  return r;
}

/*******************************************/
/*                                         */
/* process requests from wrapper functions */
/*                                         */
/*******************************************/


void process_chown(struct my_msgbuf *buf){
  struct chown_param *par;
  struct stat *st;
  set <struct stat, less<struct stat> >::iterator i;

  par=(struct chown_param *)get_par(buf);
  
  if(debug){
    fprintf(stderr,"FAKEROOT: chown, Owner=%i, Group=%i\n",
	   par->owner,par->group);
    debug_stat(&par->st);
  }
  i=data.find(par->st);
  if(i!=data.end())
    st=(struct stat *)&(*i);
  else
    st=&par->st;
  
  st->st_uid=par->owner;
  st->st_gid=par->group;
  insert_or_overwrite(data,*st);
}
void process_fchown(struct my_msgbuf *buf){
  struct fchown_param *par;
  struct stat *st;
  set <struct stat, less<struct stat> >::iterator i;

  par=(struct fchown_param *)get_par(buf);
  
  if(debug)
    fprintf(stderr,"FAKEROOT: fchown, Owner=%i, Group=%i\n",
	   par->owner,par->group);

  i=data.find(par->st);
  if(i!=data.end())
    st=(struct stat *)&(*i);
  else
    st=&par->st;

  st->st_uid=par->owner;
  st->st_gid=par->group;
  insert_or_overwrite(data,*st);
}

void process_chmod(struct my_msgbuf *buf){
  struct chmod_param *par;
  struct stat *st;
  set <struct stat, less<struct stat> >::iterator i;
  
  par=(struct chmod_param *)get_par(buf);

  if(debug)
    fprintf(stderr,"FAKEROOT: chmod, mode=%o\n",
	    par->mode);
  
  i=data.find(par->st);
  if(i!=data.end())
    st=(struct stat *)&(*i);
  else{
    st=&par->st;
    st->st_uid=0;
    st->st_gid=0;
  }
  st->st_mode = (par->mode&~S_IFMT) | (st->st_mode&S_IFMT);
  insert_or_overwrite(data,*st);
}
void process_fchmod(struct my_msgbuf *buf){
  struct fchmod_param *par;
  struct stat *st;
  set <struct stat, less<struct stat> >::iterator i;
  
  par=(struct fchmod_param *)get_par(buf);

  if(debug)
    fprintf(stderr,"FAKEROOT: chmod, mode=%o\n",
	   par->mode);

  i=data.find(par->st);
  if(i!=data.end())
    st=(struct stat *)&(*i);
  else{
    st=&par->st;
    st->st_uid=0;
    st->st_gid=0;
  }
  par->st.st_mode = (par->mode&~S_IFMT) | (st->st_mode&S_IFMT);
  insert_or_overwrite(data,*st);
}
void process_mknod(struct my_msgbuf *buf){
  struct mknod_param *par;
  struct stat *st;
  set <struct stat, less<struct stat> >::iterator i;
  
  par=(struct mknod_param *)get_par(buf);

  if(debug)
    fprintf(stderr,"FAKEROOT: mknod, mode=%o, dev=%Li\n",
	   par->mode,par->dev);

  i=data.find(par->st);
  if(i!=data.end())
    st=(struct stat *)&(*i);
  else{
    st=&par->st;
    st->st_uid=0;
    st->st_gid=0;
  }
  st->st_rdev=par->dev;
  st->st_mode=par->mode;
  insert_or_overwrite(data,*st);
  if(debug){
    fprintf(stderr,"FAKEROOT: just inserted:");
    debug_stat(st);
  }
}

void process_stat(struct my_msgbuf *buf){
  struct stat_param *par;
  set <struct stat, less<struct stat> >::iterator i;
  
  par=(struct stat_param *)get_par(buf);

  i=data.find(par->st);
  if(debug){
    fprintf(stderr,"FAKEROOT: process stat oldstate=");
    debug_stat(&par->st);
  }
  if(i==data.end()){
    if (debug)
      fprintf(stderr,"FAKEROOT:    (previously unknown)\n");
    par->st.st_uid=0;
    par->st.st_gid=0;
  }
  else{
    copy_stat(&par->st,&(*i));
    if(debug){
      fprintf(stderr,"FAKEROOT: (previously known): fake=");
      debug_stat(&par->st);      
    }

  }
  /*  
    if(par->st.st_uid>2){
    fprintf(stderr,"FAKEROOT: after stat, failing?: known=%i, stat=",
	    i==data.end());
    debug_stat(&par->st);
  }
  */
  send_stat(buf);
}
//void process_fstat(struct my_msgbuf *buf){
//  process_stat(buf);
//}

void process_unlink(struct my_msgbuf *buf){
  struct unlink_param *par;
  
  par=(struct unlink_param *)get_par(buf);


  if((par->st.st_nlink==1)||
     (S_ISDIR(par->st.st_mode)&&(par->st.st_nlink==2))){
    set <struct stat, less<struct stat> >::iterator i;
    i=data.find(par->st);
    if(i!=data.end()){
      if(debug){
	printf("FAKEROOT: unlink known file, old stat=");
	debug_stat(&(*i));
      }
      data.erase(i);
    }
    if(data.find(par->st)!=data.end()){
      fprintf(stderr,"FAKEROOT************************************************* cannot remove stat\n");
    }
  }
}
void process_checklibc(struct my_msgbuf *buf){
  struct checklibc_param *par;
  
  par=(struct checklibc_param *)get_par(buf);

  if(mixed_libc_hack)
    par->check=check_libc(par->path);
  else
    par->check=0;
  
  if(debug)
    fprintf(stderr,"FAKEROOT: checklibs, check=%i\n",par->check);
  send_msg_size(buf,sizeof(par->check));  
}

void process_debianpedantic(struct my_msgbuf *buf){
  struct debianpedantic_param *par;
  
  par=(struct debianpedantic_param *)get_par(buf);
  
  par->debian=debian_pedantic;
  send_msg_size(buf,sizeof(*par));
}

void process_debug(struct my_msgbuf *buf){
  struct debug_param *par;
  
  
  par=(struct debug_param *)get_par(buf);
  
  if(debug)
    fprintf(stderr,"libfakeroot (%i): %s\n",par->pid, par->msg);
}

void process_debugdata(struct my_msgbuf *){
  set <struct stat, less<struct stat> >::iterator i;
  
  for(i=data.begin(); i!=data.end(); i++)
    debug_stat(&(*i));
}
void process_msg(struct my_msgbuf *buf){

  func_id f;
  f=  ((struct q_and_a_func_id *)&(buf->mtext))->id;
  //fprintf(stderr,"FAKEROOT: processing msg %i\n",f);
  func_arr[f](buf);
  
}

void get_msg(){

  struct my_msgbuf buf;
  int r=0;

  if(debug)
    fprintf(stderr,"FAKEROOT: msg=%i, key=%i\n",msg_get,msg_key);
  do{
    r=msgrcv(msg_get,&buf,MAXMSG,0,0);
    if(0)
      fprintf(stderr,"FAKEROOT: r=%i, recieved message type=%li, message=%i\n",r,buf.mtype,*(func_id*)buf.mtext);
    if(r!=-1)
      process_msg(&buf);
  }while ((r!=-1)||(errno==EINTR));
  if(debug){
    perror("FAKEROOT, get_msg");
    fprintf(stderr,"r=%i, EINTR=%i\n",errno,EINTR);
  }
}

/***********/
/*         */
/* misc    */
/*         */
/***********/


int run_command(char *const*cmd){
  char *default_cmd[]={"/bin/sh",NULL,NULL};
  char buf[300];
  int status;
  pid_t child;

  if(!*cmd)
    cmd=default_cmd;

  if(debug)
    fprintf(stderr,"FAKEROOT: running %s\n", cmd[0]);
  if(!(child=fork())){

    if(!fakelib)
      setenv("LD_PRELOAD", "/usr/lib/libfakeroot/libfakeroot.so.0", 1);
    else
      setenv("LD_PRELOAD", &(fakelib[0]),1);      
    sprintf(buf,"%i",msg_key);
    setenv(FAKEROOTKEY_ENV,buf,1);

    execvp(cmd[0],cmd);/*, __environ);*/
    sprintf(buf,"fakeroot, execve %s", cmd[0]);
    perror(buf);
    exit(1);
  }
  waitpid(child, &status, 0);
  return status;
}

void cleanup(){
  union semun sem_union;
  if(debug)
    fprintf(stderr,"fakeroot: clearing up message ques and semaphores\n");
  msgctl (msg_get, IPC_RMID,NULL);
  msgctl (msg_send, IPC_RMID,NULL);
  semctl (sem_id,0,IPC_RMID,sem_union);
}

/***********/
/*         */
/* main    */
/*         */
/***********/

int main(int /*argc*/, char **argv){
  union semun sem_union;
  int status;

  if(getenv(FAKEROOTKEY_ENV)){
    //(I'm not sure -- maybe this can work?)
    fprintf(stderr,"Please, don't run fakeroot from within fakeroot!\n");
    exit(1);
  }

  srandom(time(NULL)+getpid()*33151);
  msg_key=random();
  msg_get=msgget(msg_key,IPC_CREAT|0600);
  msg_send=msgget(msg_key+1,IPC_CREAT|0600);
  sem_id=semget(msg_key+2,1,IPC_CREAT|0600);
  sem_union.val=1;
  semctl (sem_id,0,SETVAL,sem_union);

  if((msg_get==-1)||(msg_send==-1)||(sem_id==-1)){
    perror("fakeroot, while creating message channels");
    cleanup();
    exit(1);
  }
  if(debug)
    fprintf(stderr,"msg_key=%i\n",msg_key);

  while(*(++argv)){
    if(!strcmp(*argv,"-d"))
      debug=true;
    else if(!strcmp(*argv,"--lib")){
       if(*(++argv))
	 fakelib=*argv;
    }
    else if(!strcmp(*argv,"--debian"))
      debian_pedantic=true;
    else if(!strcmp(*argv,"--nodebian"))
      debian_pedantic=false;
    else if(!strcmp(*argv,"--mixedlibhack"))
      mixed_libc_hack=1;
    else if(!strcmp(*argv,"--nomixedlibhack"))
      mixed_libc_hack=0;
    else if(!strcmp(*argv,"-c")&&*(++argv)){
      break;
    }else if(*argv){
      break;
    }
  }
  if(!fork()){
    get_msg();
    exit(3);
  }
  status=run_command(argv);
  cleanup();
  return status/256;
}



