/*
 network.c : Network stuff

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../common.h"
#include "../nls.h"
#include "network.h"

#define TRANSMIT_MAX_WAIT 1000000 /* microseconds */

typedef struct
{
    time_t created;
    gint handle;
    gint tag;
}
NET_DISCONNECT_REC;

static GList *disconnects;

#ifdef HAVE_IPV6
static gint host_error;
#endif

/* wait for max. 5 seconds before forcing to close the socket */
#define MAX_CLOSE_WAIT 5

/* Cygwin need this, don't know others.. */
/*#define BLOCKING_SOCKETS 1*/

#ifndef HAVE_IPV6
/* IRIX 6.5's inet_ntoa() bugs with gcc (it says always 255.255.255.255) */
static gboolean broken_inet_ntoa = FALSE;
#endif

/* Connect to socket */
gint net_connect(gchar *addr, gint port, IPADDR *my_ip)
{
    IPADDR ip;

    g_return_val_if_fail(addr != NULL, -1);

    if (net_gethostname(addr, &ip) == -1)
        return -1;

    return net_connect_ip(&ip, port, my_ip);
}

/* Connect to socket with ip address */
gint net_connect_ip(IPADDR *ip, gint port, IPADDR *my_ip)
{
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin6;
#endif
    struct sockaddr_in sin;
    gint opt, fh, ret;

    memset(&sin, 0, sizeof(sin));
#ifdef HAVE_IPV6
    memset(&sin6, 0, sizeof(sin6));
    if (ip->family == AF_INET6)
    {
	sin6.sin6_family = ip->family;
	fh = socket(AF_INET6, SOCK_STREAM, 0);
    }
    else
#endif
    {
	sin.sin_family = ip->family;
	fh = socket(AF_INET, SOCK_STREAM, 0);
    }

    if (fh == -1) return -1;

    opt = 1;
    setsockopt(fh, SOL_SOCKET, SO_REUSEADDR, (gchar *) &opt, sizeof(opt));
    setsockopt(fh, SOL_SOCKET, SO_KEEPALIVE, (gchar *) &opt, sizeof(opt));

    fcntl(fh, F_SETFL, O_NONBLOCK);

    if (my_ip != NULL)
    {
	/* set our own address */
#ifdef HAVE_IPV6
	if (my_ip->family == AF_INET6)
	{
	    sin6.sin6_family = my_ip->family;
	    memcpy(&sin6.sin6_addr, &my_ip->addr, sizeof(my_ip->addr));
	    /* just ignore if bind() fails */
	    bind(fh, (struct sockaddr *) &sin6, sizeof(sin6));
	}
	else
#endif
	{
	    sin.sin_family = my_ip->family;
	    memcpy(&sin.sin_addr, &my_ip->addr, 4);
	    /* just ignore if bind() fails */
	    bind(fh, (struct sockaddr *) &sin, sizeof(sin));
	}
    }

#ifdef HAVE_IPV6
    if (ip->family == AF_INET6)
    {
	sin6.sin6_family = ip->family;
	memcpy(&sin6.sin6_addr, &ip->addr, sizeof(ip->addr));
	sin6.sin6_port = htons(port);
	ret = connect(fh, (struct sockaddr *) &sin6, sizeof (sin6));
    }
    else
#endif
    {
	sin.sin_family = ip->family;
	memcpy(&sin.sin_addr, &ip->addr, 4);
	sin.sin_port = htons(port);
	ret = connect(fh, (struct sockaddr *) &sin, sizeof (sin));
    }

    if (ret < 0 && errno != EINPROGRESS)
    {
        close(fh);
        return -1;
    }

    return fh;
}

/* Disconnect socket */
void net_disconnect(gint fh)
{
    g_return_if_fail(fh != -1);

    close(fh);
}

static void net_disconnect_remove(NET_DISCONNECT_REC *rec)
{
    gui_input_remove(rec->tag);
    g_free(rec);
    disconnects = g_list_remove(disconnects, rec);
}

static void sig_disconnect(NET_DISCONNECT_REC *rec)
{
    gchar buf[128];
    gint ret;

    for (;;)
    {
	ret = net_receive(rec->handle, buf, sizeof(buf));
	if (ret <= 0)
	{
	    if (ret == -1)
		net_disconnect_remove(rec);
	    break;
	}
    }
}

void net_disconnect_later(gint fh)
{
    NET_DISCONNECT_REC *rec;

    rec = g_new(NET_DISCONNECT_REC, 1);
    disconnects = g_list_append(disconnects, rec);

    rec->created = time(NULL);
    rec->handle = fh;
    rec->tag = gui_input_add(fh, GUI_INPUT_READ, (GUIInputFunction) sig_disconnect, rec);
}

/* Listen for connections on a socket */
gint net_listen(IPADDR *my_ip, gint *port)
{
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin6;
#endif
    struct sockaddr_in sin;
    gint opt, len, ret, fh;

    g_return_val_if_fail(my_ip != NULL, -1);
    g_return_val_if_fail(port != NULL, -1);

    memset(&sin, 0, sizeof(sin));
#ifdef HAVE_IPV6
    memset(&sin6, 0, sizeof(sin6));
    if (my_ip->family == AF_INET6)
    {
	sin6.sin6_family = my_ip->family;
	memcpy(&sin6.sin6_addr, &my_ip->addr, sizeof(my_ip->addr));
	sin6.sin6_port = htons(*port);
	fh = socket(AF_INET6, SOCK_STREAM, 0);
    }
    else
#endif
    {
	sin.sin_family = my_ip->family;
	memcpy(&sin.sin_addr, &my_ip->addr, sizeof(my_ip->addr));
	sin.sin_port = htons(*port);
	fh = socket(AF_INET, SOCK_STREAM, 0);
    }

    if (fh == -1) return -1;

    opt = 1;
    setsockopt(fh, SOL_SOCKET, SO_REUSEADDR, (gchar *) &opt, sizeof(opt));
    setsockopt(fh, SOL_SOCKET, SO_KEEPALIVE, (gchar *) &opt, sizeof(opt));

#ifdef HAVE_IPV6
    if (my_ip->family == AF_INET6)
	ret = bind(fh, (struct sockaddr *) &sin6, sizeof(sin6));
    else
#endif
	ret = bind(fh, (struct sockaddr *) &sin, sizeof(sin));

    if (ret < 0)
    {
        close(fh);
        return -1;
    }

#ifdef HAVE_IPV6
    if (my_ip->family == AF_INET6)
    {
	len = sizeof(sin6);
	ret = getsockname(fh, (struct sockaddr *) &sin6, &len);
    }
    else
#endif
    {
	len = sizeof(sin);
	ret = getsockname(fh, (struct sockaddr *) &sin, &len);
    }

    if (ret != 0)
    {
        close(fh);
        return -1;
    }
#ifdef HAVE_IPV6
    if (my_ip->family == AF_INET6)
	*port = ntohs(sin6.sin6_port);
    else
#endif
	*port = ntohs(sin.sin_port);

    fcntl(fh, F_SETFL, O_NONBLOCK);

    if (listen(fh, 1) < 0)
    {
        close(fh);
        return -1;
    }

    return fh;
}

/* Accept a connection on a socket */
gint net_accept(gint handle, IPADDR *addr, gint *port)
{
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin6;
    struct sockaddr_in *sin;
#else
    struct sockaddr_in sin;
#endif
    gint ret, addrlen;

    g_return_val_if_fail(handle != -1, -1);
    g_return_val_if_fail(addr != NULL, -1);
    g_return_val_if_fail(port != NULL, -1);

#ifdef HAVE_IPV6
    addrlen = sizeof(sin6);
    ret = accept(handle, (struct sockaddr *) &sin6, &addrlen);
#else
    addrlen = sizeof(sin);
    ret = accept(handle, (struct sockaddr *) &sin, &addrlen);
#endif

    if (ret < 0) return -1;

#ifdef HAVE_IPV6
    addr->family = sin6.sin6_family;
    if (addr->family == AF_INET6)
    {
	memcpy(&addr->addr, &sin6.sin6_addr, sizeof(addr->addr));
	*port = ntohs(sin6.sin6_port);
    }
    else
    {
	sin = (struct sockaddr_in *) &sin6;
	memcpy(&addr->addr, &sin->sin_addr, sizeof(addr->addr));
	*port = ntohs(sin->sin_port);
    }
#else
    addr->family = AF_INET;
    memcpy(&addr->addr, &sin.sin_addr, sizeof(addr->addr));
    *port = ntohs(sin.sin_port);
#endif

    fcntl(ret, F_SETFL, O_NONBLOCK);
    return ret;
}

/* Read data from socket */
gint net_receive(gint fh, gchar *buf, gint len)
{
    gint n;
#ifdef BLOCKING_SOCKETS
    fd_set set;
    struct timeval tv;
#endif

    g_return_val_if_fail(fh != -1, -1);
    g_return_val_if_fail(buf != NULL, -1);

#ifdef BLOCKING_SOCKETS
    FD_ZERO(&set);
    FD_SET(fh, &set);
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    if (select(fh+1, &set, NULL, NULL, &tv) <= 0 ||
        !FD_ISSET(fh, &set)) return 0;
#endif
    n = recv(fh, buf, len, 0);
    if (n == 0) return -1;
    if (n == -1 && (errno == EWOULDBLOCK || errno == EAGAIN)) return 0;
    return n;
}

/* Transmit data (blocking) */
gint net_transmit(gint fh, gchar *data, gint len)
{
    gint n, sent = 0;
    gint timeout = 0;

    g_return_val_if_fail(fh != -1, -1);
    g_return_val_if_fail(data != NULL, -1);

    do
    {
        n = send(fh, data+sent, len-sent, 0);
        if (n == 0) return -1;
        if (n == -1)
        {
            if (errno == EWOULDBLOCK || errno == EAGAIN)
	    {
		timeout += 10000;
		if (timeout > TRANSMIT_MAX_WAIT)
		    return sent;
                usleep(10000);
                continue;
            }
            return -1;
        }
        sent += n;
    }
    while (sent != len);

    return sent;
}

/* Transmit data (nonblocking) */
gint net_transmit_nonblock(gint fh, gchar *data, gint len)
{
    gint n;

    g_return_val_if_fail(fh != -1, -1);
    g_return_val_if_fail(data != NULL, -1);

    n = send(fh, data, len, 0);
    if (n == -1 && (errno == EWOULDBLOCK || errno == EAGAIN))
        return 0;

    return n > 0 ? n : -1;
}

/* Get socket address/port */
gint net_getsockname(gint handle, IPADDR *addr, gint *port)
{
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin;
    struct sockaddr_in *sin4;
#else
    struct sockaddr_in sin;
#endif
    gint len = sizeof(sin);

    g_return_val_if_fail(handle != -1, -1);
    g_return_val_if_fail(addr != NULL, -1);

    if (getsockname(handle, (struct sockaddr *) &sin, &len) == -1)
        return 0;

#ifdef HAVE_IPV6
    if (sin.sin6_family == AF_INET6)
    {
	addr->family = sin.sin6_family;
	memcpy(&addr->addr, &sin.sin6_addr, sizeof(addr->addr));
	if (port != NULL) *port = sin.sin6_port;
    }
    else
#endif
    {
#ifdef HAVE_IPV6
	sin4 = (struct sockaddr_in *) &sin;
	addr->family = sin4->sin_family;
	memcpy(&addr->addr, &sin4->sin_addr, sizeof(addr->addr));
	if (port != NULL) *port = sin4->sin_port;
#else
	addr->family = sin.sin_family;
	memcpy(&addr->addr, &sin.sin_addr, sizeof(addr->addr));
	if (port != NULL) *port = sin.sin_port;
#endif
    }

    return 1;
}

/* Get IP address for host */
gint net_gethostname(gchar *addr, IPADDR *ip)
{
#ifdef HAVE_IPV6
    struct addrinfo req, *ai;
    gchar hbuf[NI_MAXHOST];
#else
    struct hostent *hp;
#endif

    g_return_val_if_fail(addr != NULL, -1);

    /* host name */
#ifdef HAVE_IPV6
    memset(ip, 0, sizeof(IPADDR));
    memset(&req, 0, sizeof(struct addrinfo));
    req.ai_socktype = SOCK_STREAM;

    /* save error to host_error for later use */
    host_error = getaddrinfo(addr, NULL, &req, &ai);
    if (host_error != 0)
	return -1;

    if (getnameinfo(ai->ai_addr, ai->ai_addrlen, hbuf, sizeof(hbuf), NULL, 0, NI_NUMERICHOST))
    {
	host_error = 1;
	return -1;
    }

    ip->family = ai->ai_family;
    if (ai->ai_family == AF_INET6)
	memcpy(&ip->addr, &((struct sockaddr_in6 *) (ai->ai_addr))->sin6_addr, sizeof(ip->addr));
    else
	memcpy(&ip->addr, &((struct sockaddr_in *) (ai->ai_addr))->sin_addr, 4);
    freeaddrinfo(ai);
#else
    hp = gethostbyname(addr);
    if (hp == NULL) return -1;

    ip->family = AF_INET;
    memcpy(&ip->addr, hp->h_addr, 4);
#endif

    return 0;
}

gulong net_long2net(gulong num)
{
    return htonl(num);
}

gulong net_net2long(gulong num)
{
    return ntohl(num);
}

gchar *net_ip2host(IPADDR *ip)
{
#ifdef HAVE_IPV6
    static gchar str[INET6_ADDRSTRLEN];

    if (inet_ntop(ip->family, &ip->addr, str, sizeof(str)) == NULL)
	strcpy(str, "!error!");
    return str;
#else
    static gchar str[20];
    gulong ip4;

    if (!broken_inet_ntoa)
	return inet_ntoa(ip->addr);

    ip4 = net_net2long(ip->addr.s_addr);
    sprintf(str, "%lu.%lu.%lu.%lu",
	    (ip4 & 0xff000000) >> 24,
	    (ip4 & 0x00ff0000) >> 16,
	    (ip4 & 0x0000ff00) >> 8,
	    (ip4 & 0x000000ff));
    return str;
#endif
}

IPADDR *net_host2ip(gchar *host)
{
    static IPADDR ip;
    gulong addr;

#ifdef HAVE_IPV6
    if (strchr(host, ':') == NULL)
#endif
    {
	/* IPv4 */
	addr = inet_addr(host);
	ip.family = AF_INET;
	memcpy(&ip.addr, &addr, 4);
    }
#ifdef HAVE_IPV6
    else
    {
	ip.family = AF_INET6;
	if (inet_pton(AF_INET6, host, &ip.addr) == 0)
	    memset(&ip.addr, 0, sizeof(ip.addr));
    }
#endif

    return &ip;
}

gint net_geterror(gint handle)
{
    gint data, len = sizeof(data);

    if (getsockopt(handle, SOL_SOCKET, SO_ERROR, &data, &len) != 0)
        return -1;

    return data;
}

gchar *net_gethosterror(void)
{
#ifdef HAVE_IPV6
    if (host_error == 0)
    {
	/* no error?! */
	return NULL;
    }

    if (host_error == 1)
    {
	/* getnameinfo() failed ..
	   FIXME: does strerror return the right error message?? */
        return strerror(errno);
    }

    return gai_strerror(host_error);
#else
    switch (h_errno)
    {
	case HOST_NOT_FOUND:
	    return _("Host not found");
	case NO_ADDRESS:
	    return _("No IP address found for name");
	case NO_RECOVERY:
	    return _("A non-recovable name server error occurred");
	case TRY_AGAIN:
	    return _("A temporary error on an authoritative name server");
    }

    /* unknown error */
    return NULL;
#endif
}

void net_init(void)
{
#ifndef HAVE_IPV6
    IPADDR ip;
    gchar *p1, *p2;

    /* check if inet_ntoa() is broken. */
    ip.family = AF_INET;
    ip.addr.s_addr = 0x12345678;
    broken_inet_ntoa = TRUE;
    p1 = net_ip2host(&ip);
    broken_inet_ntoa = FALSE;
    p2 = net_ip2host(&ip);

    broken_inet_ntoa = strcmp(p1, p2) != 0;
#else
    host_error = 0;
#endif
}

void net_deinit(void)
{
    NET_DISCONNECT_REC *rec;
    gboolean first;
    time_t now, max;
    struct timeval tv;
    fd_set set;

    if (disconnects == NULL)
	return;

    /* give the sockets a chance to disconnect themselves.. */
    max = time(NULL)+MAX_CLOSE_WAIT;
    first = TRUE;
    while (disconnects != NULL)
    {
	rec = disconnects->data;

	now = time(NULL);
	if (rec->created+MAX_CLOSE_WAIT <= now || max <= now)
	{
	    /* this one has waited enough */
	    net_disconnect_remove(rec);
	    continue;
	}

	FD_ZERO(&set);
	FD_SET(rec->handle, &set);
	tv.tv_sec = first ? 0 : max-now;
	tv.tv_usec = first ? 100000 : 0;
	if (select(rec->handle+1, &set, NULL, NULL, &tv) > 0 && FD_ISSET(rec->handle, &set))
	{
	    /* data coming .. check if we can close the handle */
	    sig_disconnect(rec);
	}
	else
	{
	    /* Display the text when we have already waited for a while */
	    printf(_("Please wait, waiting for servers to close connections..\n"));
	    fflush(stdout);

            first = FALSE;
	}
    }
}
