/*
 snapshot.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint showtag = -1;

static void snapshot_show(WINDOW_REC *window)
{
    WINDOW_VIEW_REC *view;
    GUI_WINDOW_REC *gui;
    GtkWidget *scrollbox, *frame;
    GtkAdjustment *adj;
    gint x, y, xsize, ysize;

    g_return_if_fail(window != NULL);

    view = WINDOW_VIEW(window);
    gui = WINDOW_GUI(window);
    gdk_window_get_pointer (NULL, &x, &y, NULL);

    gui->snapshot = gtk_window_new(GTK_WINDOW_POPUP);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
    gtk_container_add(GTK_CONTAINER(gui->snapshot), frame);

    if (!view->itext)
    {
	scrollbox = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(frame), scrollbox);
    }
    else
    {
	scrollbox = frame;
    }

    xsize = view->text->allocation.width;
    ysize = view->text->allocation.height;

    if (xsize <= 1) xsize = gdk_screen_width()/2;
    if (ysize <= 1) ysize = gdk_screen_height()*2/5;
    gtk_widget_set_usize(scrollbox, xsize, ysize);

    if (x + xsize > gdk_screen_width())
        x -= xsize + 5;
    else
        x += 5;

    if (y + ysize > gdk_screen_height())
        y -= ysize + 5;
    else
        y += 5;

    gtk_widget_set_uposition(gui->snapshot, x, y);

    gtk_object_set_data(GTK_OBJECT(view->text), "parent", view->text->parent);
    gtk_widget_reparent(view->text, scrollbox);
    gtk_widget_show_all(gui->snapshot);

    adj = view->itext ?
	GTK_ITEXT(view->text)->adjustment :
	GTK_TEXT(view->text)->vadj;
    gtk_adjustment_set_value(adj, adj->upper - adj->lower - adj->page_size);
}

static void snapshot_hide(WINDOW_REC *window)
{
    WINDOW_VIEW_REC *view;
    GUI_WINDOW_REC *gui;
    GtkWidget *parent;
    gint oldpage, page;

    g_return_if_fail(window != NULL);

    view = WINDOW_VIEW(window);
    gui = WINDOW_GUI(window);

    parent = gtk_object_get_data(GTK_OBJECT(view->text), "parent");
    g_return_if_fail(parent != NULL);

    gtk_widget_reparent(view->text, parent);
    if (view->itext)
	gtk_box_reorder_child(GTK_BOX(parent), view->text, 0);
    gtk_widget_destroy(gui->snapshot);
    gui->snapshot = NULL;

    if (g_list_length(gui->parent->children) > 1)
    {
        /* the page comes back over the active window, not in the real window
           it belongs to, just switch to some other window for a while and it
           works again. */
        oldpage = gtk_notebook_page_num(GTK_NOTEBOOK(gui->parent->notebook), WINDOW_GUI(gui->parent->active)->window);

        page = gtk_notebook_page_num(GTK_NOTEBOOK(gui->parent->notebook), gui->window);
        gtk_notebook_set_page(GTK_NOTEBOOK(gui->parent->notebook), GPOINTER_TO_INT(page));
        gtk_notebook_set_page(GTK_NOTEBOOK(gui->parent->notebook), GPOINTER_TO_INT(oldpage));
    }
}

static gint show_timeout(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, 0);

    snapshot_show(CHANNEL_PARENT(channel));
    showtag = -1;
    return 0;
}

/* enter_notify_event */
gint snapshot_sig_show_channel(GtkWidget *widget, GdkEventCrossing *event, CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, 0);

    if (showtag == -1)
        showtag = gui_timeout_add(500, (GUITimeoutFunction) show_timeout, channel);

    return 0;
}

/* leave_notify_event */
gint snapshot_sig_hide_channel(GtkWidget *widget, GdkEventCrossing *event, CHANNEL_REC *channel)
{
    g_return_val_if_fail(widget != NULL, 0);
    g_return_val_if_fail(channel != NULL, 0);

    if (showtag != -1)
    {
        gui_timeout_remove(showtag);
        showtag = -1;
    }
    else if (WINDOW_GUI(CHANNEL_PARENT(channel))->snapshot != NULL)
    {
        snapshot_hide(CHANNEL_PARENT(channel));
        if (gtk_object_get_data(GTK_OBJECT(widget), "open_window"))
        {
            signal_emit("gui channel open", 1, channel);
            gtk_object_set_data(GTK_OBJECT(widget), "open_window", GINT_TO_POINTER(FALSE));
        }
    }

    return 0;
}

/* signal: button_press_event */
gint snapshot_sig_open_channel(GtkWidget *widget, GdkEventButton *event, CHANNEL_REC *channel)
{
    g_return_val_if_fail(widget != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(channel != NULL, 0);

    if (event->button == 3)
    {
        gui_popupmenu_create(channel, NULL, event);
        return 1;
    }

    /* only left button opens channel */
    if (event->button != 1) return 0;

    if (showtag != -1)
    {
        gui_timeout_remove(showtag);
        showtag = -1;
    }
    else if (WINDOW_GUI(CHANNEL_PARENT(channel))->snapshot != NULL)
    {
        /* calling hide_snapshot() here causes some weird bug .. */
        gtk_object_set_data(GTK_OBJECT(widget), "open_window", GINT_TO_POINTER(TRUE));
        return 0;
    }

    signal_emit("gui channel open", 1, channel);
    return 0;
}
