/*
 dialog-irssibot.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gchar *nicks = "irssibot irssibot2 crasb";
static gchar *irssibot_masks[] =
{
    "*!cras@*.sieme.net",
    "*!cras@fun157.kivikko.hoas.fi",
};

static GtkWidget *startup_dialog;
static time_t last_check;
static gchar *last_stable, *last_unstable;

static gboolean check_stable, check_unstable;

void sig_destroyed(void)
{
    proplist_t iprop;

    /* set stable version to ? so we don't get this notice again. */
    iprop = config_section(&cprop, "irssibot");
    config_set_str(iprop, "last_stable", "?");
    config_set_bool(iprop, "check_stable", check_stable);
    config_set_bool(iprop, "check_unstable", check_unstable);

    PLSave(cprop, TRUE);
    startup_dialog = NULL;
}

void sig_notifynew_clicked(GtkToggleButton *button, GtkWidget *widget)
{
    check_stable = gtk_toggle_button_get_active(button);
    gtk_widget_set_sensitive(widget, check_stable);
}

void sig_notify_unstable_clicked(GtkToggleButton *button)
{
    check_unstable = gtk_toggle_button_get_active(button);
}

void dialog_irssibot_config(void)
{
    proplist_t iprop;
    GtkWidget *dialog, *label, *checkbox, *checkbox2;

    startup_dialog = dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, NULL);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "delete_event",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
		       GTK_SIGNAL_FUNC(sig_destroyed), NULL);
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

    label = gtk_label_new(_("Irssi can inform you when new versions are\n"
			    "available for download, you just need to be\n"
			    "connected to some irc network where irssibot\n"
			    "is, like ircnet, efnet or gimpnet."));

    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 10);

    checkbox = gtk_check_button_new_with_label(_("Notify about new irssi releases"));
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), checkbox, FALSE, FALSE, 0);

    checkbox2 = gtk_check_button_new_with_label(_("Notify also about new development releases"));
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), checkbox2, FALSE, FALSE, 0);

    gtk_signal_connect(GTK_OBJECT(checkbox), "clicked",
		       GTK_SIGNAL_FUNC(sig_notifynew_clicked), checkbox2);
    gtk_signal_connect(GTK_OBJECT(checkbox2), "clicked",
		       GTK_SIGNAL_FUNC(sig_notify_unstable_clicked), NULL);

    iprop = config_get_prop(cprop, "irssibot");
    check_stable = config_get_bool(iprop, "check_stable", TRUE);
    check_unstable = config_get_bool(iprop, "check_unstable", FALSE);

    if (check_stable)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), TRUE);
    if (check_unstable)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox2), TRUE);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_widget_show_all(dialog);
}

static gboolean sig_raise(void)
{
    if (startup_dialog != NULL)
	gdk_window_raise(startup_dialog->window);
    return TRUE;
}

static void sig_get_irssi(GtkWidget *widget, gchar *version)
{
    SERVER_REC *server;
    gchar *nick, *str;

    server = gtk_object_get_data(GTK_OBJECT(widget), "server");
    nick = gtk_object_get_data(GTK_OBJECT(widget), "nick");

    str = g_strdup_printf("PRIVMSG %s :\001IRSSI_DCC_SEND_%s\001", nick, version);
    irc_send_cmd(server, str);
    g_free(str);
}

static void sig_open_website(void)
{
    gchar *str;

    str = g_strdup_printf(setup_get_str("url_www_client"), IRSSI_WEBSITE);
    execute(str);
    g_free(str);
}

static void dialog_irssibot_new_version(SERVER_REC *server, gchar *nick, gchar *stable, gchar *unstable)
{
    GtkWidget *dialog, *label, *button, *buttonbox;
    gboolean stable_changed, unstable_changed;
    gchar *str;

    stable_changed = *stable != '\0' && (last_stable == NULL || strcmp(stable, last_stable) != 0);
    unstable_changed = *unstable != '\0' && (check_unstable && (last_unstable == NULL || strcmp(unstable, last_unstable) != 0));

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    if (stable_changed && unstable_changed)
	str = g_strdup_printf(_("Both stable and unstable Irssi versions\nhave been released.\n\nStable version %s\nUnstable version %s"), stable, unstable);
    else if (stable_changed)
	str = g_strdup_printf(_("New stable irssi version %s\nhas been released."), stable);
    else
	str = g_strdup_printf(_("New unstable irssi version %s\nhas been released."), unstable);

    label = gtk_label_new(str);
    g_free(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0);

    buttonbox = gtk_vbutton_box_new();
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 7);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), buttonbox, FALSE, FALSE, 0);

    nick = g_strdup(nick);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
			      GTK_SIGNAL_FUNC(g_free), (GtkObject *) nick);

    if (stable_changed)
    {
	button = gtk_button_new_with_label (_("Get stable Irssi via DCC"));
	gtk_object_set_data(GTK_OBJECT(button), "server", server);
	gtk_object_set_data(GTK_OBJECT(button), "nick", nick);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_get_irssi), "STABLE");
	gtk_container_add(GTK_CONTAINER(buttonbox), button);
    }
    if (unstable_changed)
    {
	button = gtk_button_new_with_label (_("Get unstable Irssi via DCC"));
	gtk_object_set_data(GTK_OBJECT(button), "server", server);
	gtk_object_set_data(GTK_OBJECT(button), "nick", nick);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_get_irssi), "UNSTABLE");
	gtk_container_add(GTK_CONTAINER(buttonbox), button);
    }
    button = gtk_button_new_with_label (_("Open Irssi's website"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_open_website), NULL);
    gtk_container_add(GTK_CONTAINER(buttonbox), button);

    gtk_widget_show_all(dialog);
}

static gboolean sig_irssiversion(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    proplist_t iprop;
    gchar *params, *timestamp, *stable, *unstable;
    gint n;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    /* verify that it's irssi bot where we get this ctcp .. */
    for (n = 0; n < sizeof(irssibot_masks)/sizeof(irssibot_masks[0]); n++)
    {
	if (irc_mask_match_address(irssibot_masks[n], nick, addr))
	{
	    /* yep, it's irssi bot! */
	    n = -1;
	    break;
	}
    }

    if (n != -1)
    {
	/* nope, irssi bot isn't sending this CTCP. */
	signal_emit("default ctcp msg", 5, data, server, nick, addr, target);
	return TRUE;
    }

    params = cmd_get_params(data, 3, &timestamp, &stable, &unstable);
    if (last_stable != NULL && strcmp(stable, last_stable) == 0 &&
	(!check_unstable || (last_unstable != NULL && strcmp(unstable, last_unstable) == 0)))
    {
	/* nothing new */
	g_free(params);
	return TRUE;
    }

    dialog_irssibot_new_version(server, nick, stable, unstable);

    /* save the versions to configuration file */
    iprop = config_section(&cprop, "irssibot");
    config_set_str(iprop, "last_stable", stable);
    config_set_str(iprop, "last_unstable", unstable);
    PLSave(cprop, TRUE);

    if (last_stable != NULL) g_free(last_stable);
    last_stable = g_strdup(stable);
    if (last_unstable != NULL) g_free(last_unstable);
    last_unstable = g_strdup(unstable);

    g_free(params);
    return TRUE;
}

static void check_irssi(SERVER_REC *server)
{
    gchar *str;

    /* check only max. once in every 12 hours */
    if ((!check_stable && !check_unstable) || last_check+(60*60*12) > time(NULL))
	return;

    /* Check if there's new irssi version */
    str = g_strdup_printf("USERHOST %s", nicks);
    irc_send_cmd(server, str);
    g_free(str);

    /* need to redirect it so ui-common doesn't print anything.. */
    server_redirect_event(server, nicks, 1,
			  "event 302", "irssibot userhost", -1,
			  "event 401", "event empty", 1, NULL);
}

static gboolean sig_userhost(gchar *data, SERVER_REC *server)
{
    GList *addrs, *tmp;
    gchar *params, *hosts, *nick, *ptr, *str;
    gint n;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &hosts);

    if (*hosts == '\0')
    {
	g_free(params);
	return TRUE;
    }

    nick = NULL;
    addrs = str2list(hosts, ' ');
    for (tmp = addrs; tmp != NULL; tmp = tmp->next)
    {
        ptr = strchr((gchar *) tmp->data, '=');
        if (ptr == NULL) continue;
	*ptr = '\0'; ptr += 2;

	for (n = 0; n < sizeof(irssibot_masks)/sizeof(irssibot_masks[0]); n++)
	{
	    if (irc_mask_match_address(irssibot_masks[n], tmp->data, ptr))
	    {
		/* yep, it's irssi bot! */
		nick = tmp->data;
		break;
	    }
	}
    }

    if (nick != NULL)
    {
	/* send IRSSI_VERSION CTCP to irssi bot */
	str = g_strdup_printf("PRIVMSG %s :\001IRSSI_VERSION\001", nick);
	irc_send_cmd(server, str);
	g_free(str);
    }

    if (addrs != NULL)
    {
	g_free(addrs->data);
	g_list_free(addrs);
    }
    g_free(params);
    return TRUE;
}

static gboolean sig_connected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server_redirect_init(server, "command userhost", 1, "event 302", "event 401", NULL);
    check_irssi(server);
    return TRUE;
}

void dialog_irssibot_init(void)
{
    proplist_t iprop;
    gchar *value;

    iprop = config_get_prop(cprop, "irssibot");

    /* get the last irssi versions.. */
    last_check = 0;
    value = config_get_str(iprop, "last_stable", NULL);
    last_stable = value == NULL ? NULL : g_strdup(value);
    value = config_get_str(iprop, "last_unstable", NULL);
    last_unstable = value == NULL ? NULL : g_strdup(value);

    startup_dialog = NULL;
    if (last_stable == NULL)
    {
	/* first time this feature is being used, display a config dialog .. */
	dialog_irssibot_config();
	check_stable = TRUE;
	check_unstable = FALSE;
    }
    else
    {
	check_stable = config_get_bool(iprop, "check_stable", TRUE);
	check_unstable = config_get_bool(iprop, "check_unstable", FALSE);
    }

    signal_add("event connected", (SIGNAL_FUNC) sig_connected);
    signal_add("irssibot userhost", (SIGNAL_FUNC) sig_userhost);
    signal_add("ctcp reply irssi_version", (SIGNAL_FUNC) sig_irssiversion);
    signal_add("gui mainwindow created", (SIGNAL_FUNC) sig_raise);
}

void dialog_irssibot_deinit(void)
{
    if (last_stable != NULL) g_free(last_stable);
    if (last_unstable != NULL) g_free(last_unstable);

    signal_remove("event connected", (SIGNAL_FUNC) sig_connected);
    signal_remove("irssibot userhost", (SIGNAL_FUNC) sig_userhost);
    signal_remove("ctcp reply irssi_version", (SIGNAL_FUNC) sig_irssiversion);
    signal_remove("gui mainwindow created", (SIGNAL_FUNC) sig_raise);
}
