<?php
namespace OpenTelemetry\Context {
use DDTrace\SpanData;
use DDTrace\Tag;
use DDTrace\Util\ObjectKVStore;
use OpenTelemetry\API\Trace as API;
use OpenTelemetry\API\Baggage\Baggage;
use OpenTelemetry\API\Baggage\BaggageBuilder;
use OpenTelemetry\SDK\Common\Attribute\Attributes;
use OpenTelemetry\SDK\Common\Attribute\AttributesFactory;
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeFactory;
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeInterface;
use OpenTelemetry\SDK\Resource\ResourceInfoFactory;
use OpenTelemetry\SDK\Trace as SDK;
use OpenTelemetry\SDK\Trace\SpanProcessor\NoopSpanProcessor;
use function DDTrace\active_span;
use function DDTrace\generate_distributed_tracing_headers;
use function spl_object_id;
final class Context implements ContextInterface
{
    private static ContextStorageInterface $storage;
    private static string $storageClass = '';
    private static ContextKeyInterface $spanContextKey;
    private static ContextKeyInterface $baggageContextKey;
    private ?object $span = null;
    private array $context = [];
    private array $contextKeys = [];
    private function __construct()
    {
        self::$spanContextKey = ContextKeys::span();
        self::$baggageContextKey = ContextKeys::baggage();
    }
    public static function createKey(string $key): ContextKeyInterface
    {
        return new ContextKey($key);
    }
    public static function setStorage(ContextStorageInterface $storage): void
    {
        self::$storage = $storage;
    }
    public static function storage(): ContextStorageInterface
    {
        if (self::$storageClass === '') {
            self::$storageClass = class_exists('OpenTelemetry\Context\FiberBoundContextStorageExecutionAwareBC') ? 'OpenTelemetry\Context\FiberBoundContextStorageExecutionAwareBC' : 'OpenTelemetry\Context\ContextStorage';
        }
        return self::$storage ??= new self::$storageClass();
    }
    public static function resolve($context, ?ContextStorageInterface $contextStorage = null): ContextInterface
    {
        $spanFromContext = API\Span::fromContext(self::storage()->current());
        if ($spanFromContext instanceof SDK\Span) {
            $ddSpanFromContext = $spanFromContext->getDDSpan();
            self::deactivateEndedParents($ddSpanFromContext);
        }
        self::activateParent(active_span());
        return $context ?? ($contextStorage ?? self::storage())->current() ?: self::getRoot();
    }
    public static function getRoot(): ContextInterface
    {
        static $empty;
        return $empty ??= new self();
    }
    private static function getDDInstrumentationScope(): InstrumentationScopeInterface
    {
        static $instrumentationScope;
        return $instrumentationScope ??= (new InstrumentationScopeFactory(new AttributesFactory()))->create('datadog');
    }
    public static function getCurrent(): ContextInterface
    {
        $spanFromContext = API\Span::fromContext(self::storage()->current());
        if ($spanFromContext instanceof SDK\Span) {
            $ddSpanFromContext = $spanFromContext->getDDSpan();
            self::deactivateEndedParents($ddSpanFromContext);
        }
        return self::activateParent(active_span());
    }
    private static function deactivateEndedParents(?SpanData $currentSpan)
    {
        if ($currentSpan === null) {
            return;
        }
        if ($currentSpan->getDuration() === 0) {
            return;
        }
        $OTelCurrentSpan = ObjectKVStore::get($currentSpan, 'otel_span');
        if ($OTelCurrentSpan !== null) {
            $OTelCurrentSpan->endOTelSpan();
        }
        self::deactivateEndedParents($currentSpan->parent);
    }
    private static function convertDDSpanKindToOtel(string $spanKind)
    {
        switch ($spanKind) {
            case Tag::SPAN_KIND_VALUE_CLIENT:
                return API\SpanKind::KIND_CLIENT;
            case Tag::SPAN_KIND_VALUE_SERVER:
                return API\SpanKind::KIND_SERVER;
            case Tag::SPAN_KIND_VALUE_PRODUCER:
                return API\SpanKind::KIND_PRODUCER;
            case Tag::SPAN_KIND_VALUE_CONSUMER:
                return API\SpanKind::KIND_CONSUMER;
            case Tag::SPAN_KIND_VALUE_INTERNAL:
            default:
                return API\SpanKind::KIND_INTERNAL;
        }
    }
    private static function activateParent(?SpanData $currentSpan): ContextInterface
    {
        if ($currentSpan === null) {
            return self::storage()->current();
        }
        $OTelCurrentSpan = ObjectKVStore::get($currentSpan, 'otel_span');
        if ($OTelCurrentSpan !== null) {
            if (ObjectKVStore::get($currentSpan, 'ddtrace_scope_activated')) {
                return self::storage()->current()->with(self::$spanContextKey, $OTelCurrentSpan);
            } else {
                return self::storage()->current();
            }
        }
        $parentContext = self::activateParent($currentSpan->parent);
        $currentSpanId = $currentSpan->hexId();
        $currentTraceId = \DDTrace\root_span()->traceId;
        $traceContext = generate_distributed_tracing_headers(['tracecontext']);
        $traceFlags = isset($traceContext['traceparent']) ? substr($traceContext['traceparent'], -2) === '01' ? API\TraceFlags::SAMPLED : API\TraceFlags::DEFAULT : null;
        $traceState = new API\TraceState($traceContext['tracestate'] ?? null);
        $links = [];
        foreach ($currentSpan->links as $spanLink) {
            $linkSpanContext = API\SpanContext::create($spanLink->traceId, $spanLink->spanId, API\TraceFlags::DEFAULT, new API\TraceState($spanLink->traceState ?? null));
            $links[] = new SDK\Link($linkSpanContext, Attributes::create($spanLink->attributes ?? []));
        }
        $events = [];
        foreach ($currentSpan->events as $spanEvent) {
            $events[] = new SDK\Event($spanEvent->name, (int) $spanEvent->timestamp, Attributes::create((array) $spanEvent->attributes ?? []));
        }
        $OTelCurrentSpan = SDK\Span::startSpan(
            $currentSpan,
            API\SpanContext::create($currentTraceId, $currentSpanId, $traceFlags, $traceState),
            self::getDDInstrumentationScope(),
            isset($currentSpan->meta[Tag::SPAN_KIND]) ? self::convertDDSpanKindToOtel($currentSpan->meta[Tag::SPAN_KIND]) : API\SpanKind::KIND_INTERNAL,
            API\Span::fromContext($parentContext),
            $parentContext,
            NoopSpanProcessor::getInstance(),
            ResourceInfoFactory::emptyResource(),
            [],
            $links,
            count($links),
            $events,
            false
        );
        ObjectKVStore::put($currentSpan, 'otel_span', $OTelCurrentSpan);
        $currentContext = $parentContext->with(self::$spanContextKey, $OTelCurrentSpan);
        ObjectKVStore::put($currentSpan, 'ddtrace_scope_activated', true);
        self::storage()->attach($currentContext);
        return $currentContext;
    }
    public function activate(): ScopeInterface
    {
        if ($this->span instanceof SDK\Span) {
            ObjectKVStore::put($this->span->getDDSpan(), 'ddtrace_scope_activated', true);
        }
        $scope = self::storage()->attach($this);
        return $scope;
    }
    public function withContextValue(ImplicitContextKeyedInterface $value): ContextInterface
    {
        return $value->storeInContext($this);
    }
    public function with(ContextKeyInterface $key, $value): self
    {
        if ($this->get($key) === $value) {
            return $this;
        }
        $self = clone $this;
        if ($key === self::$spanContextKey) {
            $self->span = $value;
            return $self;
        }
        $id = spl_object_id($key);
        if ($key === self::$baggageContextKey) {
            if ($this->span instanceof SDK\Span) {
                $currentDdSpan = $self->span ? $self->span->getDDSpan() : null;
                if ($currentDdSpan !== null) {
                    $currentDdSpan->baggage = [];
                    foreach ($value->getAll() as $baggageKey => $baggageEntry) {
                        $currentDdSpan->baggage[$baggageKey] = $baggageEntry->getValue();
                    }
                    return $self;
                }
            }
        }
        if ($value !== null) {
            $self->context[$id] = $value;
            $self->contextKeys[$id] ??= $key;
        } else {
            unset($self->context[$id], $self->contextKeys[$id]);
        }
        return $self;
    }
    public function get(ContextKeyInterface $key)
    {
        if ($key === self::$spanContextKey) {
            return $this->span;
        }
        if ($key === self::$baggageContextKey) {
            if ($this->span instanceof SDK\Span) {
                $currentDdSpan = $this->span ? $this->span->getDDSpan() : null;
                if ($currentDdSpan) {
                    $baggageBuilder = new BaggageBuilder();
                    foreach ($currentDdSpan->baggage as $baggageKey => $baggageValue) {
                        $baggageBuilder->set($baggageKey, $baggageValue);
                    }
                    return $baggageBuilder->build();
                }
            }
        }
        return $this->context[spl_object_id($key)] ?? null;
    }
}
}
namespace DDTrace\OpenTelemetry {
use DDTrace\SpanData;
use DDTrace\Tag;
class Convention
{
    public static function defaultOperationName(SpanData $span): string
    {
        $meta = $span->meta;
        $spanKind = $meta[Tag::SPAN_KIND] ?? null;
        switch (true) {
            case isset($meta['http.request.method']) && $spanKind === Tag::SPAN_KIND_VALUE_SERVER:
                return 'http.server.request';
            case isset($meta['http.request.method']) && $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return 'http.client.request';
            case isset($meta['db.system']) && $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return strtolower($meta['db.system']) . '.query';
            case isset($meta['messaging.system'], $meta['messaging.operation']) && in_array($spanKind, [Tag::SPAN_KIND_VALUE_CONSUMER, Tag::SPAN_KIND_VALUE_PRODUCER, Tag::SPAN_KIND_VALUE_SERVER, Tag::SPAN_KIND_VALUE_CLIENT]):
                return strtolower($meta['messaging.system']) . '.' . strtolower($meta['messaging.operation']);
            case isset($meta['rpc.system']) && $meta['rpc.system'] === 'aws-api' && $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return isset($meta['rpc.service']) ? 'aws.' . strtolower($meta['rpc.service']) . '.request' : 'aws.client.request';
            case isset($meta['rpc.system']) && $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return strtolower($meta['rpc.system']) . '.client.request';
            case isset($meta['rpc.system']) && $spanKind === Tag::SPAN_KIND_VALUE_SERVER:
                return strtolower($meta['rpc.system']) . '.server.request';
            case isset($meta['faas.trigger']) && $spanKind === Tag::SPAN_KIND_VALUE_SERVER:
                return strtolower($meta['faas.trigger']) . '.invoke';
            case isset($meta['faas.invoked_provider'], $meta['faas.invoked_name']) && $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return strtolower($meta['faas.invoked_provider']) . '.' . strtolower($meta['faas.invoked_name']) . '.invoke';
            case isset($meta['graphql.operation.type']):
                return 'graphql.server.request';
            case $spanKind === Tag::SPAN_KIND_VALUE_SERVER:
            case $spanKind === Tag::SPAN_KIND_VALUE_CLIENT:
                return isset($meta['network.protocol.name']) ? strtolower($meta['network.protocol.name']) . ".{$spanKind}.request" : "{$spanKind}.request";
            case !empty($spanKind):
                return $spanKind;
            default:
                return '';
        }
    }
}
}
namespace DDTrace\OpenTelemetry\API\Trace {
use DDTrace\OpenTelemetry\SDK\Trace\Span;
use DDTrace\SpanData;
use OpenTelemetry\API\Trace as API;
use OpenTelemetry\API\Trace\SpanContextInterface;
use OpenTelemetry\API\Trace\SpanContextValidator;
use OpenTelemetry\API\Trace\TraceFlags;
use OpenTelemetry\API\Trace\TraceState;
use OpenTelemetry\API\Trace\TraceStateInterface;
use function DDTrace\generate_distributed_tracing_headers;
final class SpanContext implements SpanContextInterface
{
    private $span;
    private bool $sampled;
    private bool $remote;
    private ?string $traceId;
    private ?string $spanId;
    private bool $isValid = true;
    private ?string $currentTracestateString = null;
    private ?TraceStateInterface $currentTracestateInstance = null;
    private function __construct(SpanData $span, bool $sampled, bool $remote, ?string $traceId = null, ?string $spanId = null)
    {
        $this->span = $span;
        $this->sampled = $sampled;
        $this->remote = $remote;
        $this->traceId = $traceId ?: \DDTrace\root_span()->traceId;
        $this->spanId = $spanId ?: $this->span->hexId();
        if (!SpanContextValidator::isValidTraceId($this->traceId) || !SpanContextValidator::isValidSpanId($this->spanId)) {
            $this->traceId = SpanContextValidator::INVALID_TRACE;
            $this->spanId = SpanContextValidator::INVALID_SPAN;
            $this->isValid = false;
        }
    }
    public function getTraceId(): string
    {
        return $this->traceId;
    }
    public function getTraceIdBinary(): string
    {
        return hex2bin($this->getTraceId());
    }
    public function getSpanId(): string
    {
        return $this->spanId;
    }
    public function getSpanIdBinary(): string
    {
        return hex2bin($this->getSpanId());
    }
    public function getTraceState(): ?TraceStateInterface
    {
        $current = \DDTrace\active_stack();
        if ($current !== $this->span->stack) {
            \DDTrace\switch_stack($this->span);
            $traceContext = generate_distributed_tracing_headers(['tracecontext']);
            \DDTrace\switch_stack($current);
        } else {
            $traceContext = generate_distributed_tracing_headers(['tracecontext']);
        }
        $newTracestate = $traceContext['tracestate'] ?? null;
        if ($this->currentTracestateInstance === null || $this->currentTracestateString !== $newTracestate) {
            $this->currentTracestateString = $newTracestate;
            $this->currentTracestateInstance = new TraceState($newTracestate);
        }
        return $this->currentTracestateInstance;
    }
    public function isSampled(): bool
    {
        return $this->sampled;
    }
    public function isValid(): bool
    {
        return $this->isValid;
    }
    public function isRemote(): bool
    {
        return $this->remote;
    }
    public function getTraceFlags(): int
    {
        return $this->sampled ? TraceFlags::SAMPLED : TraceFlags::DEFAULT;
    }
    public static function createFromRemoteParent(string $traceId, string $spanId, int $traceFlags = TraceFlags::DEFAULT, ?TraceStateInterface $traceState = null): SpanContextInterface
    {
        return API\SpanContext::createFromRemoteParent($traceId, $spanId, $traceFlags, $traceState);
    }
    public static function create(string $traceId, string $spanId, int $traceFlags = TraceFlags::DEFAULT, ?TraceStateInterface $traceState = null): SpanContextInterface
    {
        return API\SpanContext::create($traceId, $spanId, $traceFlags, $traceState);
    }
    public static function getInvalid(): SpanContextInterface
    {
        return API\SpanContext::getInvalid();
    }
    public static function createFromLocalSpan(SpanData $span, bool $sampled, ?string $traceId = null, ?string $spanId = null)
    {
        return new self($span, $sampled, false, $traceId, $spanId);
    }
}
}
namespace OpenTelemetry\SDK\Trace {
use DDTrace\SpanData;
use DDTrace\SpanLink;
use DDTrace\SpanEvent;
use DDTrace\ExceptionSpanEvent;
use DDTrace\Tag;
use DDTrace\OpenTelemetry\Convention;
use DDTrace\Util\ObjectKVStore;
use OpenTelemetry\API\Trace as API;
use OpenTelemetry\SDK\Trace\LinkInterface;
use OpenTelemetry\API\Trace\SpanContextInterface;
use OpenTelemetry\API\Trace\SpanInterface;
use OpenTelemetry\Context\ContextInterface;
use OpenTelemetry\SDK\Common\Attribute\Attributes;
use OpenTelemetry\SDK\Common\Attribute\AttributesBuilderInterface;
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeInterface;
use OpenTelemetry\SDK\Common\Time\ClockFactory;
use OpenTelemetry\SDK\Resource\ResourceInfo;
use Throwable;
use function DDTrace\close_span;
use function DDTrace\switch_stack;
use function DDTrace\Internal\add_span_flag;
final class Span extends API\Span implements ReadWriteSpanInterface
{
    private SpanData $span;
    private API\SpanContextInterface $context;
    private API\SpanContextInterface $parentSpanContext;
    private SpanProcessorInterface $spanProcessor;
    private array $links;
    private int $totalRecordedLinks;
    private array $events;
    private int $totalRecordedEvents;
    private int $kind;
    private ResourceInfo $resource;
    private InstrumentationScopeInterface $instrumentationScope;
    private StatusDataInterface $status;
    private string $operationNameConvention = "";
    private function __construct(SpanData $span, API\SpanContextInterface $context, InstrumentationScopeInterface $instrumentationScope, int $kind, API\SpanContextInterface $parentSpanContext, SpanProcessorInterface $spanProcessor, ResourceInfo $resource, array $links = [], int $totalRecordedLinks = 0, array $events = [], bool $isRemapped = true)
    {
        $this->span = $span;
        $this->context = $context;
        $this->instrumentationScope = $instrumentationScope;
        $this->kind = $kind;
        $this->parentSpanContext = $parentSpanContext;
        $this->spanProcessor = $spanProcessor;
        $this->resource = $resource;
        $this->links = $links;
        $this->totalRecordedLinks = $totalRecordedLinks;
        $this->events = $events;
        $this->status = StatusData::unset();
        if ($isRemapped || empty($span->name)) {
            $span->name = $this->operationNameConvention = Convention::defaultOperationName($span);
        }
        if ($isRemapped) {
            foreach ($links as $link) {
                $linkContext = $link->getSpanContext();
                $span->links[] = $this->createAndSaveSpanLink($linkContext, $link->getAttributes()->toArray(), $link);
            }
            foreach ($events as $event) {
                $spanEvent = new SpanEvent($event->getName(), $event->getAttributes()->toArray(), $event->getEpochNanos());
                ObjectKVStore::put($spanEvent, "event", $event);
                $span->events[] = $spanEvent;
            }
        }
    }
    public static function startSpan(SpanData $span, API\SpanContextInterface $context, InstrumentationScopeInterface $instrumentationScope, int $kind, API\SpanInterface $parentSpan, ContextInterface $parentContext, SpanProcessorInterface $spanProcessor, ResourceInfo $resource, array $attributes, array $links, int $totalRecordedLinks, array $events, bool $isRemapped = true): self
    {
        self::_setAttributes($span, $attributes);
        $resourceAttributes = $resource->getAttributes()->toArray();
        self::_setAttributes($span, $resourceAttributes);
        add_span_flag($span, \DDTrace\Internal\SPAN_FLAG_OPENTELEMETRY);
        $OTelSpan = new self($span, $context, $instrumentationScope, $kind, $parentSpan->getContext(), $spanProcessor, $resource, $links, $totalRecordedLinks, $events, $isRemapped);
        ObjectKVStore::put($span, 'otel_span', $OTelSpan);
        $spanProcessor->onStart($OTelSpan, $parentContext);
        return $OTelSpan;
    }
    public function getName(): string
    {
        return $this->span->resource ?: $this->span->name;
    }
    public function getContext(): SpanContextInterface
    {
        return $this->context;
    }
    public function getParentContext(): API\SpanContextInterface
    {
        return $this->parentSpanContext;
    }
    public function getInstrumentationScope(): InstrumentationScopeInterface
    {
        return $this->instrumentationScope;
    }
    public function hasEnded(): bool
    {
        return $this->span->getDuration() !== 0;
    }
    public function toSpanData(): SpanDataInterface
    {
        $hasEnded = $this->hasEnded();
        $this->updateSpanLinks();
        $this->updateSpanEvents();
        if (method_exists(SpanInterface::class, 'addLink')) {
            return new ImmutableSpan($this, $this->getName(), $this->links, $this->events, Attributes::create(array_merge($this->span->meta, $this->span->metrics)), $this->totalRecordedEvents, $this->totalRecordedLinks, StatusData::create($this->status->getCode(), $this->status->getDescription()), $hasEnded ? $this->span->getStartTime() + $this->span->getDuration() : 0, $this->hasEnded());
        } else {
            return new ImmutableSpan($this, $this->getName(), $this->links, $this->events, Attributes::create(array_merge($this->span->meta, $this->span->metrics)), $this->totalRecordedEvents, StatusData::create($this->status->getCode(), $this->status->getDescription()), $hasEnded ? $this->span->getStartTime() + $this->span->getDuration() : 0, $this->hasEnded());
        }
    }
    public function getDuration(): int
    {
        return $this->span->getDuration() ?: ClockFactory::getDefault()->now() - $this->span->getStartTime();
    }
    public function getKind(): int
    {
        return $this->kind;
    }
    public function getAttribute(string $key)
    {
        return $this->span->meta[$key] ?? $this->span->metrics[$key] ?? null;
    }
    public function getStartEpochNanos(): int
    {
        return $this->span->getStartTime();
    }
    public function getTotalRecordedLinks(): int
    {
        return $this->totalRecordedLinks;
    }
    public function getTotalRecordedEvents(): int
    {
        return $this->totalRecordedEvents;
    }
    public function isRecording(): bool
    {
        return !$this->hasEnded();
    }
    private static function _setAttribute(SpanData $span, string $key, $value): void
    {
        if ($value === null) {
            unset($span->meta[$key]);
            unset($span->metrics[$key]);
        } elseif ($key[0] === '_' && \strncmp($key, '_dd.p.', 6) === 0) {
            $distributedKey = \substr($key, 6);
            \DDTrace\add_distributed_tag($distributedKey, $value);
        } elseif (\is_float($value) || \is_int($value) || \is_array($value) && \count($value) > 0 && \is_numeric($value[0])) {
            $span->metrics[$key] = $value;
        } elseif ($key === 'service.name') {
            $span->service = $value;
        } else {
            $span->meta[$key] = $value;
        }
    }
    private static function _setAttributes(SpanData $span, iterable $attributes): void
    {
        foreach ($attributes as $key => $value) {
            self::_setAttribute($span, $key, $value);
        }
    }
    private function updateConvention(): void
    {
        if ($this->span->name === $this->operationNameConvention) {
            $this->span->name = $this->operationNameConvention = Convention::defaultOperationName($this->span);
        }
    }
    public function setAttribute(string $key, $value): SpanInterface
    {
        if (!$this->hasEnded()) {
            self::_setAttribute($this->span, $key, $value);
        }
        $this->updateConvention();
        return $this;
    }
    public function setAttributes(iterable $attributes): SpanInterface
    {
        if (!$this->hasEnded()) {
            foreach ($attributes as $key => $value) {
                $this->setAttribute($key, $value);
            }
        }
        $this->updateConvention();
        return $this;
    }
    public function addLink(SpanContextInterface $context, iterable $attributes = []): SpanInterface
    {
        if ($this->hasEnded() || !$context->isValid()) {
            return $this;
        }
        $this->span->links[] = $this->createAndSaveSpanLink($context, $attributes);
        return $this;
    }
    public function addEvent(string $name, iterable $attributes = [], $timestamp = null): SpanInterface
    {
        if (!$this->hasEnded()) {
            $this->span->events[] = new SpanEvent($name, $attributes, $timestamp ?? (int) (microtime(true) * 1000000000.0));
        }
        return $this;
    }
    public function recordException(Throwable $exception, iterable $attributes = []): SpanInterface
    {
        if (!$this->hasEnded()) {
            $this->setAttribute(Tag::ERROR_STACK, \DDTrace\get_sanitized_exception_trace($exception));
            $this->span->events[] = new ExceptionSpanEvent($exception, $attributes);
        }
        return $this;
    }
    public function updateName(string $name): SpanInterface
    {
        if (!$this->hasEnded()) {
            $this->span->resource = $name;
        }
        return $this;
    }
    public function setStatus(string $code, ?string $description = null): SpanInterface
    {
        if ($this->hasEnded()) {
            return $this;
        }
        if ($code === API\StatusCode::STATUS_UNSET) {
            return $this;
        }
        if ($this->status->getCode() === API\StatusCode::STATUS_OK) {
            return $this;
        }
        if ($this->status->getCode() === API\StatusCode::STATUS_UNSET && $code === API\StatusCode::STATUS_ERROR) {
            $this->span->meta[Tag::ERROR_MSG] = $description;
        } elseif ($this->status->getCode() === API\StatusCode::STATUS_ERROR && $code === API\StatusCode::STATUS_OK) {
            unset($this->span->meta[Tag::ERROR_MSG]);
            unset($this->span->meta[Tag::ERROR_TYPE]);
            unset($this->span->meta[Tag::ERROR_STACK]);
        }
        $this->status = StatusData::create($code, $description);
        return $this;
    }
    public function end(?int $endEpochNanos = null): void
    {
        if ($this->hasEnded()) {
            return;
        }
        $this->endOTelSpan($endEpochNanos);
        switch_stack($this->span);
        close_span($endEpochNanos !== null ? $endEpochNanos / 1000000000 : 0);
        $this->spanProcessor->onEnd($this);
    }
    public function endOTelSpan(?int $endEpochNanos = null): void
    {
        if ($this->hasEnded()) {
            return;
        }
        if (empty($this->span->name)) {
            $this->span->name = Convention::defaultOperationName($this->span);
        }
        $this->context = API\SpanContext::create($this->context->getTraceId(), $this->context->getSpanId(), $this->context->getTraceFlags(), $this->context->getTraceState());
    }
    public function getResource(): ResourceInfo
    {
        return $this->resource;
    }
    public function getDDSpan(): SpanData
    {
        return $this->span;
    }
    private function updateSpanLinks()
    {
        $datadogSpanLinks = $this->span->links;
        $otel = [];
        foreach ($datadogSpanLinks as $datadogSpanLink) {
            $link = ObjectKVStore::get($datadogSpanLink, "link");
            if ($link === null) {
                $link = new Link(API\SpanContext::create($datadogSpanLink->traceId, $datadogSpanLink->spanId, $this->context->getTraceFlags(), new API\TraceState($datadogSpanLink->traceState ?? null)), Attributes::create($datadogSpanLink->attributes ?? []));
                ObjectKVStore::put($datadogSpanLink, "link", $link);
            }
            $otel[] = $link;
        }
        $this->links = $otel;
        $this->totalRecordedLinks = count($otel);
    }
    private function updateSpanEvents()
    {
        $datadogSpanEvents = $this->span->events;
        $this->span->meta["events"] = count($this->events);
        $otel = [];
        foreach ($datadogSpanEvents as $datadogSpanEvent) {
            $exceptionAttributes = [];
            $event = ObjectKVStore::get($datadogSpanEvent, "event");
            if ($event === null) {
                if ($datadogSpanEvent instanceof ExceptionSpanEvent) {
                    $exceptionAttributes = ['exception.message' => $attributes['exception.message'] ?? $datadogSpanEvent->exception->getMessage(), 'exception.type' => $attributes['exception.type'] ?? get_class($datadogSpanEvent->exception), 'exception.stacktrace' => $attributes['exception.stacktrace'] ?? \DDTrace\get_sanitized_exception_trace($datadogSpanEvent->exception)];
                }
                $event = new Event($datadogSpanEvent->name, (int) $datadogSpanEvent->timestamp, Attributes::create(array_merge($exceptionAttributes, \is_array($datadogSpanEvent->attributes) ? $datadogSpanEvent->attributes : iterator_to_array($datadogSpanEvent->attributes))));
                ObjectKVStore::put($datadogSpanEvent, "event", $event);
            }
            $otel[] = $event;
        }
        $this->events = $otel;
        $this->totalRecordedEvents = count($otel);
    }
    private function createAndSaveSpanLink(SpanContextInterface $context, iterable $attributes = [], ?LinkInterface $link = null)
    {
        $spanLink = new SpanLink();
        $spanLink->traceId = $context->getTraceId();
        $spanLink->spanId = $context->getSpanId();
        $spanLink->traceState = (string) $context->getTraceState();
        $spanLink->attributes = $attributes;
        $spanLink->droppedAttributesCount = 0;
        if (is_null($link)) {
            $link = new Link($context, Attributes::create($attributes));
        }
        ObjectKVStore::put($spanLink, "link", $link);
        return $spanLink;
    }
}
}
namespace OpenTelemetry\SDK\Trace {
use DDTrace\OpenTelemetry\API\Trace as DDTraceAPI;
use DDTrace\Tag;
use OpenTelemetry\API\Trace as API;
use OpenTelemetry\API\Trace\SpanBuilderInterface;
use OpenTelemetry\API\Trace\SpanContextInterface;
use OpenTelemetry\API\Trace\SpanInterface;
use OpenTelemetry\Context\Context;
use OpenTelemetry\Context\ContextKeys;
use OpenTelemetry\Context\ContextInterface;
use OpenTelemetry\SDK\Common\Attribute\Attributes;
use OpenTelemetry\SDK\Common\Attribute\AttributesFactory;
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeInterface;
use OpenTelemetry\SDK\Resource\ResourceInfoFactory;
use Throwable;
final class SpanBuilder implements API\SpanBuilderInterface
{
    private string $spanName;
    private InstrumentationScopeInterface $instrumentationScope;
    private TracerSharedState $tracerSharedState;
    private $parentContext = null;
    private int $spanKind = API\SpanKind::KIND_INTERNAL;
    private array $links = [];
    private array $events = [];
    private array $attributes;
    private int $totalNumberOfLinksAdded = 0;
    private float $startEpochNanos = 0;
    public function __construct(string $spanName, InstrumentationScopeInterface $instrumentationScope, TracerSharedState $tracerSharedState)
    {
        $this->spanName = $spanName;
        $this->instrumentationScope = $instrumentationScope;
        $this->tracerSharedState = $tracerSharedState;
        $this->attributes = [];
    }
    public function setParent($context): API\SpanBuilderInterface
    {
        $this->parentContext = $context;
        return $this;
    }
    public function addLink(SpanContextInterface $context, iterable $attributes = []): SpanBuilderInterface
    {
        if (!$context->isValid()) {
            return $this;
        }
        $this->totalNumberOfLinksAdded++;
        $this->links[] = new Link($context, $this->tracerSharedState->getSpanLimits()->getLinkAttributesFactory()->builder($attributes)->build());
        return $this;
    }
    public function addEvent(string $name, iterable $attributes = [], ?int $timestamp = null): SpanBuilderInterface
    {
        $this->events[] = new Event($name, $timestamp ?? (int) (microtime(true) * 1000000000.0), $this->tracerSharedState->getSpanLimits()->getEventAttributesFactory()->builder($attributes)->build());
        return $this;
    }
    public function recordException(Throwable $exception, iterable $attributes = []): SpanBuilderInterface
    {
        $exceptionAttributes = ['exception.message' => $attributes['exception.message'] ?? $exception->getMessage(), 'exception.type' => $attributes['exception.type'] ?? get_class($exception), 'exception.stacktrace' => $attributes['exception.stacktrace'] ?? \DDTrace\get_sanitized_exception_trace($exception)];
        $this->setAttribute(Tag::ERROR_STACK, $exceptionAttributes['exception.stacktrace']);
        $allAttributes = array_merge($exceptionAttributes, \is_array($attributes) ? $attributes : iterator_to_array($attributes));
        $this->addEvent('exception', $allAttributes);
        return $this;
    }
    public function setAttribute(string $key, $value): API\SpanBuilderInterface
    {
        $this->attributes[$key] = $value;
        return $this;
    }
    public function setAttributes(iterable $attributes): API\SpanBuilderInterface
    {
        foreach ($attributes as $key => $value) {
            $this->attributes[$key] = $value;
        }
        return $this;
    }
    public function setStartTimestamp(int $timestampNanos): SpanBuilderInterface
    {
        if ($timestampNanos >= 0) {
            $this->startEpochNanos = $timestampNanos / 1000000000;
        }
        return $this;
    }
    public function setSpanKind(int $spanKind): SpanBuilderInterface
    {
        $this->spanKind = $spanKind;
        return $this;
    }
    public function startSpan(): SpanInterface
    {
        $this->applySpanKind();
        $parentContext = Context::resolve($this->parentContext);
        $parentSpan = Span::fromContext($parentContext);
        $parentSpanContext = $parentSpan->getContext();
        $span = $parentSpanContext->isValid() ? null : \DDTrace\start_trace_span($this->startEpochNanos);
        $traceId = $parentSpanContext->isValid() ? $parentSpanContext->getTraceId() : \DDTrace\root_span()->traceId;
        $samplingResult = $this->tracerSharedState->getSampler()->shouldSample($parentContext, $traceId, $this->spanName, $this->spanKind, Attributes::create($this->attributes), $this->links, $this->events);
        $span = $span ?? \DDTrace\start_trace_span($this->startEpochNanos);
        $samplingDecision = $samplingResult->getDecision();
        $sampled = SamplingResult::RECORD_AND_SAMPLE === $samplingDecision;
        $samplingResultTraceState = $samplingResult->getTraceState();
        if ($parentSpanContext->isValid()) {
            $parentId = $parentSpanContext->getSpanId();
            $traceFlags = $sampled ? '01' : '00';
            $traceParent = "00-{$traceId}-{$parentId}-{$traceFlags}";
            \DDTrace\consume_distributed_tracing_headers(['traceparent' => $traceParent, 'tracestate' => (string) $samplingResultTraceState]);
        } elseif ($samplingResultTraceState) {
            $samplingResultTraceState = $samplingResultTraceState->without('dd');
            \DDTrace\root_span()->tracestate = (string) $samplingResultTraceState;
        }
        $hexSpanId = $span->hexId();
        $spanContext = DDTraceAPI\SpanContext::createFromLocalSpan($span, $sampled, $traceId, $hexSpanId);
        if (!in_array($samplingDecision, [SamplingResult::RECORD_AND_SAMPLE, SamplingResult::RECORD_ONLY], true)) {
            return Span::wrap($spanContext);
        }
        $span->resource = $this->spanName;
        $attributes = $samplingResult->getAttributes();
        foreach ($attributes as $key => $value) {
            $this->attributes[$key] = $value;
        }
        $parentSpanContextBaggage = $parentContext->get(ContextKeys::baggage());
        if ($parentSpanContextBaggage) {
            foreach ($parentSpanContextBaggage->getAll() as $baggageKey => $baggageEntry) {
                $span->baggage[$baggageKey] = $baggageEntry->getValue();
            }
        }
        return Span::startSpan($span, $spanContext, $this->instrumentationScope, $this->spanKind, $parentSpan, $parentContext, $this->tracerSharedState->getSpanProcessor(), $parentSpanContext->isValid() ? ResourceInfoFactory::emptyResource() : $this->tracerSharedState->getResource(), $this->attributes, $this->links, $this->totalNumberOfLinksAdded, $this->events);
    }
    private function applySpanKind(): void
    {
        switch ($this->spanKind) {
            case API\SpanKind::KIND_CLIENT:
                $this->setAttribute(Tag::SPAN_KIND, Tag::SPAN_KIND_VALUE_CLIENT);
                break;
            case API\SpanKind::KIND_SERVER:
                $this->setAttribute(Tag::SPAN_KIND, Tag::SPAN_KIND_VALUE_SERVER);
                break;
            case API\SpanKind::KIND_PRODUCER:
                $this->setAttribute(Tag::SPAN_KIND, Tag::SPAN_KIND_VALUE_PRODUCER);
                break;
            case API\SpanKind::KIND_CONSUMER:
                $this->setAttribute(Tag::SPAN_KIND, Tag::SPAN_KIND_VALUE_CONSUMER);
                break;
            case API\SpanKind::KIND_INTERNAL:
                $this->setAttribute(Tag::SPAN_KIND, Tag::SPAN_KIND_VALUE_INTERNAL);
                break;
            default:
                break;
        }
    }
}
}
namespace {
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeInterface;
use OpenTelemetry\SDK\Common\InstrumentationScope\Configurator;
\DDTrace\install_hook('OpenTelemetry\API\Instrumentation\CachedInstrumentation::__construct', null, function () {
    dd_trace_internal_fn("mark_integration_loaded", $this->name, $this->version);
});
\DDTrace\install_hook('OpenTelemetry\API\Instrumentation\CachedInstrumentation::tracer', null, function (\DDTrace\HookData $hook) {
    $tracer = $hook->returned;
    $name = $this->name;
    if (strpos($name, "io.opentelemetry.contrib.php.") === 0) {
        $name = substr($name, strlen("io.opentelemetry.contrib.php."));
    }
    $name = "otel.{$name}";
    $hook->overrideReturnValue(new class($tracer, $name) implements \OpenTelemetry\API\Trace\TracerInterface
    {
        public $tracer;
        public $name;
        public function __construct($tracer, $name)
        {
            $this->tracer = $tracer;
            $this->name = $name;
        }
        public function spanBuilder(string $spanName): \OpenTelemetry\API\Trace\SpanBuilderInterface
        {
            $spanBuilder = $this->tracer->spanBuilder($spanName);
            $spanBuilder->setAttribute("component", $this->name);
            return $spanBuilder;
        }
        public function isEnabled(): bool
        {
            return $this->tracer->isEnabled();
        }
        public function getInstrumentationScope(): InstrumentationScopeInterface
        {
            if (!method_exists($this->tracer, 'getInstrumentationScope')) {
                throw new \Error("There is no getInstrumentationScope method available for " . get_class($this->tracer));
            }
            return $this->tracer->getInstrumentationScope();
        }
        public function updateConfig(Configurator $configurator): void
        {
            $this->tracer->updateConfig($configurator);
        }
    });
});
}
